import React, { useState, useEffect, useMemo, useCallback, memo } from "react";
import apiService, { storage } from "../../services/apiservices";
import { Container, Grid, Card, CardMedia, Dialog, IconButton, Box, Skeleton } from "@mui/material";
import { motion } from "framer-motion";
import { ArrowBack, ArrowForward, Close, ZoomIn } from "@mui/icons-material";

const PhotoGallery = memo(() => {
  const [isOpen, setIsOpen] = useState(false);
  const [photoIndex, setPhotoIndex] = useState(0);
  const [loaded, setLoaded] = useState([]);
  const [hoverIndex, setHoverIndex] = useState(null);
  const [images, setImages] = useState([]);
  const [error, setError] = useState(null);

  useEffect(() => {
    const fetchData = async () => {
      try {
        const response = await apiService.get("galleries");
        setImages(response.data);
        setLoaded(Array(response.data.length).fill(false));
      } catch (err) {
        setError("Failed to fetch images");
      }
    };
    fetchData();
  }, []);

  useEffect(() => {
    images.forEach((src, index) => {
      const img = new Image();
      img.src = `${storage}/${src}`;
      img.onload = () => {
        setLoaded((prev) => {
          const newLoaded = [...prev];
          newLoaded[index] = true;
          return newLoaded;
        });
      };
    });
  }, [images]);

  const handleOpen = useCallback((index) => {
    setPhotoIndex(index);
    setIsOpen(true);
  }, []);

  const handleClose = useCallback(() => setIsOpen(false), []);
  const prevImage = useCallback(() => setPhotoIndex((prev) => (prev + images.length - 1) % images.length), [images.length]);
  const nextImage = useCallback(() => setPhotoIndex((prev) => (prev + 1) % images.length), [images.length]);

  const handleMouseEnter = useCallback((index) => setHoverIndex(index), []);
  const handleMouseLeave = useCallback(() => setHoverIndex(null), []);

  const galleryItems = useMemo(() => {
    return images.map((src, index) => (
      <Grid item xs={6} sm={6} md={3} key={index}>
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: index * 0.1 }}
          onMouseEnter={() => handleMouseEnter(index)}
          onMouseLeave={handleMouseLeave}
          style={{ position: "relative", cursor: "pointer" }}
        >
          <Card onClick={() => handleOpen(index)}>
            {!loaded[index] ? (
              <Skeleton variant="rectangular" height={200} />
            ) : (
              <>
                <CardMedia
                  component="img"
                  height="200"
                  image={`${storage}/${src}`}
                  alt={`Gallery ${index + 1}`}
                  loading="lazy"
                />
                {hoverIndex === index && (
                  <motion.div
                    initial={{ opacity: 0 }}
                    animate={{ opacity: 1 }}
                    transition={{ duration: 0.3 }}
                    style={{
                      position: "absolute",
                      top: 0,
                      left: 0,
                      width: "100%",
                      height: "100%",
                      background: "rgba(0, 0, 0, 0.4)",
                      backdropFilter: "blur(5px)",
                      display: "flex",
                      justifyContent: "center",
                      alignItems: "center",
                      borderRadius: "4px",
                    }}
                  >
                    <ZoomIn sx={{ fontSize: 50, color: "white" }} />
                  </motion.div>
                )}
              </>
            )}
          </Card>
        </motion.div>
      </Grid>
    ));
  }, [images, loaded, hoverIndex, handleOpen, handleMouseEnter, handleMouseLeave]);

  const lightboxContent = useMemo(() => (
    <Dialog fullScreen open={isOpen} onClose={handleClose}>
      <Box
        display="flex"
        justifyContent="center"
        alignItems="center"
        height="100vh"
        position="relative"
        bgcolor="black"
      >
        <IconButton
          onClick={handleClose}
          sx={{ position: "absolute", top: 20, right: 20, color: "white" }}
        >
          <Close sx={{ fontSize: 30 }} />
        </IconButton>

        <IconButton
          onClick={prevImage}
          sx={{ position: "absolute", left: 20, color: "white" }}
        >
          <ArrowBack sx={{ fontSize: 40 }} />
        </IconButton>

        <motion.img
          key={photoIndex}
          src={`${storage}/${images[photoIndex]}`}
          alt="Lightbox"
          initial={{ opacity: 0, scale: 0.8 }}
          animate={{ opacity: 1, scale: 1 }}
          transition={{ duration: 0.5, ease: "easeOut" }}
          style={{
            maxWidth: "90%",
            maxHeight: "90vh",
            objectFit: "contain",
            borderRadius: "8px",
          }}
        />

        <IconButton
          onClick={nextImage}
          sx={{ position: "absolute", right: 20, color: "white" }}
        >
          <ArrowForward sx={{ fontSize: 40 }} />
        </IconButton>
      </Box>
    </Dialog>
  ), [isOpen, photoIndex, images, handleClose, prevImage, nextImage]);

  return (
    <div>
      <div>
        <div className="subbaner-gallery">
          <div className="text-center breadcrumbs">
            <h4>Gallery</h4>
          </div>
        </div>
      </div>

      <Container className="mt-4 mb-4">
        <h2 className="greeting-title gallery-title text-center text-dark mb-4">
          GLIMPSES OF <span> PREVIOUS EDITION </span>
        </h2>
        <Grid container spacing={3} className="justify-content-center">
          {images.length > 0 ? galleryItems : (
            <div className="no-testimonials">
              <p>Loading.....</p>
            </div>
          )}
        </Grid>
        {lightboxContent}
      </Container>
    </div>
  );
});


export default React.memo(PhotoGallery);