import { useEffect, useState, useMemo, useCallback } from "react";
import { Swiper, SwiperSlide } from "swiper/react";
import "swiper/css";
import "swiper/css/navigation";
import "swiper/css/autoplay";
import { Navigation, Autoplay } from "swiper/modules";
import apiService, { storage } from "../../services/apiservices";

export default function BannerSlider() {
  const [banners, setBanners] = useState([]);
  const [error, setError] = useState(null);

  const fetchData = useCallback(async () => {
    try {
      const response = await apiService.get("exhibitors");

      if (response.data && typeof response.data === "object") {
        // Convert object values to an array
        const bannerArray = Object.values(response.data);
        setBanners(bannerArray);
      } else {
        setError("Invalid data format.");
      }
    } catch (err) {
      console.error("Error fetching banners:", err);
      setError("Failed to load images.");
    }
  }, []);

  useEffect(() => {
    fetchData();
  }, [fetchData]);

  // Memoize the Swiper component and slides to prevent unnecessary re-renders
  const swiperComponent = useMemo(() => {
    if (banners.length === 0) {
      return <p className="text-center">Loading images...</p>;
    }

    return (
      <Swiper
        modules={[Navigation, Autoplay]}
        navigation
        loop={true}
        autoplay={{ delay: 6000 }}
        className="mySwiper"
      >
        {banners.map((image, index) => (
          <SwiperSlide key={`${image}-${index}`}>
            <img
              src={`${storage}/${image}`}
              alt={`Slide ${index + 1}`}
              className="w-100 slide-image"
              loading="lazy" // Add lazy loading
            />
          </SwiperSlide>
        ))}
      </Swiper>
    );
  }, [banners]);

  return (
    <div className="banner-slider-main position-relative">
      <div className="row d-flex justify-content-center banner-slider-main-row">
        <div className="col-12 banner-slider-sub">
          {error && <p className="text-danger text-center">{error}</p>}
          {swiperComponent}
        </div>
      </div>
    </div>
  );
}