import React, { useEffect, useState } from "react";
import { motion } from "framer-motion";
import axios from "axios";

const SparklingFlippingText = () => {
  const [flipped, setFlipped] = useState(false);
  const [timeLeft, setTimeLeft] = useState({
    days: 0,
    hours: 0,
    minutes: 0,
    seconds: 0,
  });
  const [eventDate, setEventDate] = useState(null);
  const [event, setEvent] = useState(null);
  const [statusMessage, setStatusMessage] = useState("Loading...");

  useEffect(() => {
    // Fetch events from API
    axios
      .get("https://buildintec.codissia.com/buildintec-backend/public/api/events")
      .then((res) => {
        if (res.data.status === "success" && res.data.data.length > 0) {
          const eventData = res.data.data[0]; // take the first event
          setEvent(eventData);

          // Parse event_date (only one parse needed)
          let parsedDates = [];
          try {
            parsedDates = JSON.parse(eventData.event_date);
          } catch (err) {
            console.error("Error parsing event_date:", err);
          }

          if (parsedDates.length > 0) {
            const selectedDate = parsedDates[0].trim(); // pick first date
            const eventTime = eventData.event_time || "00:00:00"; // default midnight
            const finalDate = new Date(`${selectedDate}T${eventTime}`);
            setEventDate(finalDate);
          }
        } else {
          setStatusMessage("No Event Posted");
        }
      })
      .catch((err) => {
        console.error("API error:", err);
        setStatusMessage("No Event Posted");
      });
  }, []);

  useEffect(() => {
    if (!eventDate) return;

    setFlipped(true);

    const timer = setInterval(() => {
      const now = new Date();
      const diff = eventDate - now;

      if (diff <= 0) {
        clearInterval(timer);
        setStatusMessage("Event Already Started");
        setTimeLeft({ days: 0, hours: 0, minutes: 0, seconds: 0 });
      } else {
        setStatusMessage(""); // clear status when countdown is running
        setTimeLeft({
          days: Math.floor(diff / (1000 * 60 * 60 * 24)),
          hours: Math.floor((diff / (1000 * 60 * 60)) % 24),
          minutes: Math.floor((diff / 1000 / 60) % 60),
          seconds: Math.floor((diff / 1000) % 60),
        });
      }
    }, 1000);

    return () => clearInterval(timer);
  }, [eventDate]);

  const flipVariants = {
    hidden: { rotateX: -90, opacity: 0 },
    visible: { rotateX: 0, opacity: 1, transition: { duration: 0.6 } },
  };

  return (
    <div className="text-center ">
      {/* Event Title */}
      {event && !statusMessage && (
        <motion.h4
          className="fw-bold mb-4"
          initial={{ opacity: 0, y: -20 }}
          animate={{ opacity: flipped ? 1 : 0, y: flipped ? 0 : -20 }}
          transition={{ duration: 1 }}
        >
          {event.title} - Get Ready!
        </motion.h4>
      )}

      {/* Status Message */}
      {statusMessage && (
        <h5 className="text-danger fw-bold mb-4">{statusMessage}</h5>
      )}

      {/* Countdown Timer */}
      {!statusMessage && (
        <div className="d-flex justify-content-center gap-3 flex-wrap">
          {Object.entries(timeLeft).map(([unit, value]) => (
            <motion.div
              key={unit}
              className="counter-box d-flex flex-column align-items-center"
              variants={flipVariants}
              initial="hidden"
              animate="visible"
            >
              <span className="counter-number">{value}</span>
              <span className="counter-label text-uppercase">{unit}</span>
            </motion.div>
          ))}
        </div>
      )}
    </div>
  );
};

export default React.memo(SparklingFlippingText);
