import { motion } from "framer-motion";
import React, { useState, useEffect, useMemo } from "react";
import apiService, { storage } from "../../services/apiservices";

const HorizontalScroll = () => {
  const [images, setImages] = useState([]);
  const [error, setError] = useState(null);
  const [isMobile, setIsMobile] = useState(() => window.innerWidth <= 768);

  // Memoize the doubled images array
  const doubledImages = useMemo(() => [...images, ...images], [images]);

  useEffect(() => {
    const handleResize = () => {
      setIsMobile(window.innerWidth <= 768);
    };
    
    const resizeObserver = new ResizeObserver(handleResize);
    resizeObserver.observe(document.body);
    
    return () => resizeObserver.unobserve(document.body);
  }, []);

  useEffect(() => {
    let isMounted = true;
    
    const fetchData = async () => {
      try {
        const response = await apiService.get("galleries");
        if (isMounted) {
          setImages(response.data);
        }
      } catch (err) {
        console.error(err);
        if (isMounted) setError("Failed to fetch images");
      }
    };
    
    fetchData();
    
    return () => {
      isMounted = false;
    };
  }, []);

  if (error) return <p className="text-danger">{error}</p>;
  if (images.length === 0) return <div className="placeholder-content" />;

  return (
    <div className="unique-scroll-container d-flex flex-column align-items-center overflow-hidden">
      {/* First Scrolling Row */}
      <div className="unique-scroll-wrapper d-flex w-100 overflow-hidden">
        <motion.div
          className="unique-scroll-content d-flex"
          animate={{ x: ["0%", "-10%"] }}
          transition={{
            repeat: Infinity,
            duration: 20,
            ease: "linear",
          }}
          style={{ display: "flex", whiteSpace: "nowrap", width: "max-content" }}
        >
          {doubledImages.map((img, index) => (
            <LazyImage 
              key={`first-${index}`}
              src={`${storage}/${img}`}
              alt={`Event ${index + 15}`}
              className="unique-scroll-item mx-2"
            />
          ))}
        </motion.div>
      </div>

      {/* Second Scrolling Row (Opposite Direction) */}
      <div className="unique-scroll-wrapper d-flex w-100 overflow-hidden mt-3">
        <motion.div
          className="unique-scroll-content d-flex"
          animate={{ x: ["-10%", "0%"] }}
          transition={{
            repeat: Infinity,
            duration: 20,
            ease: "linear",
          }}
          style={{ display: "flex", whiteSpace: "nowrap", width: "max-content" }}
        >
          {doubledImages.map((img, index) => (
            <LazyImage 
              key={`second-${index}`}
              src={`${storage}/${img}`}
              alt={`Event ${index + 15}`}
              className="unique-scroll-item mx-2"
            />
          ))}
        </motion.div>
      </div>
    </div>
  );
};

// Optimized LazyImage component
const LazyImage = React.memo(({ src, alt, className }) => {
  const [loaded, setLoaded] = useState(false);
  
  return (
    <div className={className} style={{ flexShrink: 0 }}>
      {!loaded && <div className="image-placeholder" style={{ width: '200px', height: '150px' }} />}
      <img 
        className={`img-fluid ${loaded ? 'visible' : 'hidden'}`}
        src={src}
        alt={alt}
        loading="lazy"
        onLoad={() => setLoaded(true)}
        style={{ width: '200px', height: '150px', objectFit: 'cover' }}
      />
    </div>
  );
});

export default React.memo(HorizontalScroll);