import React, { useState, useEffect, useMemo, useCallback } from "react";
import { motion } from "framer-motion";
import { ArrowBack, ArrowForward, Close, ZoomIn } from "@mui/icons-material";
import apiService, { storage } from "../../services/apiservices";
import gallery from "../../assets/home/gallery/gallery.svg";
import { Grid, Card, CardMedia, Typography, Dialog, IconButton, Box, Skeleton } from "@mui/material";
import { gsap } from "gsap";
import { ScrollTrigger } from "gsap/ScrollTrigger";

gsap.registerPlugin(ScrollTrigger);

const PhotoGallery = React.memo(() => {
  const [isOpen, setIsOpen] = useState(false);
  const [photoIndex, setPhotoIndex] = useState(0);
  const [loaded, setLoaded] = useState([]);
  const [hoverIndex, setHoverIndex] = useState(null);
  const [images, setImages] = useState([]);
  const [error, setError] = useState(null);

  useEffect(() => {
    const fetchData = async () => {
      try {
        const response = await apiService.get("galleries");
        const limitedImages = response.data.slice(0, 8);
        setImages(limitedImages);
        setLoaded(Array(limitedImages.length).fill(false));
      } catch (err) {
        setError("Failed to fetch images");
      }
    };
    fetchData();
  }, []);

  useEffect(() => {
    images.forEach((src, index) => {
      const img = new Image();
      img.src = `${storage}/${src}`;
      img.onload = () => {
        setLoaded((prev) => {
          const newLoaded = [...prev];
          newLoaded[index] = true;
          return newLoaded;
        });
      };
    });
  }, [images]);

  const handleOpen = useCallback((index) => {
    setPhotoIndex(index);
    setIsOpen(true);
  }, []);

  const handleClose = useCallback(() => setIsOpen(false), []);
  const prevImage = useCallback(() => setPhotoIndex((photoIndex + images.length - 1) % images.length), [photoIndex, images.length]);
  const nextImage = useCallback(() => setPhotoIndex((photoIndex + 1) % images.length), [photoIndex, images.length]);

  const handleMouseEnter = useCallback((index) => setHoverIndex(index), []);
  const handleMouseLeave = useCallback(() => setHoverIndex(null), []);

  const imageGridItems = useMemo(() => {
    return images.map((src, index) => (
      <Grid item xs={6} sm={6} md={3} key={index}>
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: index * 0.1 }}
          onMouseEnter={() => handleMouseEnter(index)}
          onMouseLeave={handleMouseLeave}
          style={{ position: "relative", cursor: "pointer" }}
        >
          <Card onClick={() => handleOpen(index)}>
            {!loaded[index] ? (
              <Skeleton variant="rectangular" height={300} />
            ) : (
              <>
                <CardMedia
                  component="img"
                  sx={{
                    height: { xs: 200, sm: 250, md: 300 },
                    objectFit: "cover",
                  }}
                  image={`${storage}/${src}`}
                  alt={`Gallery ${index + 1}`}
                  loading="lazy"
                />
                {hoverIndex === index && (
                  <motion.div
                    initial={{ opacity: 0 }}
                    animate={{ opacity: 1 }}
                    transition={{ duration: 0.3 }}
                    style={{
                      position: "absolute",
                      top: 0,
                      left: 0,
                      width: "100%",
                      height: "100%",
                      background: "rgba(0, 126, 98, 0.25)",
                      backdropFilter: "blur(5px)",
                      display: "flex",
                      justifyContent: "center",
                      alignItems: "center",
                    }}
                  >
                    <ZoomIn sx={{ fontSize: 50, color: "white" }} />
                  </motion.div>
                )}
              </>
            )}
          </Card>
        </motion.div>
      </Grid>
    ));
  }, [images, loaded, hoverIndex, handleOpen, handleMouseEnter, handleMouseLeave]);

  const lightboxContent = useMemo(() => (
    <Dialog fullScreen open={isOpen} onClose={handleClose}>
      <Box display="flex" justifyContent="center" alignItems="center" height="100vh" position="relative" bgcolor="black">
        <IconButton onClick={handleClose} sx={{ position: "absolute", top: 20, right: 20, color: "white" }}>
          <Close sx={{ fontSize: 30 }} />
        </IconButton>
        <IconButton onClick={prevImage} sx={{ position: "absolute", left: 20, color: "white" }}>
          <ArrowBack sx={{ fontSize: 40 }} />
        </IconButton>
        <motion.img
          key={photoIndex}
          src={`${storage}/${images[photoIndex]}`}
          alt="Lightbox"
          initial={{ opacity: 0, scale: 0.8 }}
          animate={{ opacity: 1, scale: 1 }}
          transition={{ duration: 0.5, ease: "easeOut" }}
          style={{
            maxWidth: "90%",
            maxHeight: "90vh",
            objectFit: "contain",
            borderRadius: "8px",
          }}
        />
        <IconButton onClick={nextImage} sx={{ position: "absolute", right: 20, color: "white" }}>
          <ArrowForward sx={{ fontSize: 40 }} />
        </IconButton>
      </Box>
    </Dialog>
  ), [isOpen, photoIndex, images, handleClose, prevImage, nextImage]);

  return (
    <div className="gallery-section container-fluid">
      <div className="gallery-section-main text-center">
        <div className="gallery-section-img">
          <img className="img-fluid" src={gallery} alt="Gallery Banner" loading="lazy" />
        </div>
        <Typography variant="h4" className="gallery-title mt-3">
          PHOTO <strong>GALLERY</strong>
        </Typography>

        <Grid container spacing={0.7} className="justify-content-center mt-4 mb-4">
          {images.length > 0 ? (
            imageGridItems
          ) : error ? (
            <div className="no-images">
              <p>{error}</p>
            </div>
          ) : (
            <div className="no-images">
              <p>Loading images...</p>
            </div>
          )}
        </Grid>

        {lightboxContent}
      </div>
    </div>
  );
});

export default React.memo(PhotoGallery);