import { motion } from "framer-motion";
import React, { useState, useEffect, useMemo } from "react";
import apiService, { storage } from "../../services/apiservices";

const VerticalScroll = () => {
  const [isMobile, setIsMobile] = useState(() => window.innerWidth <= 768);
  const [images, setImages] = useState([]);
  const [error, setError] = useState(null);
  const [loading, setLoading] = useState(true);

  // Memoize the doubled images array
  const doubledImages = useMemo(() => [...images, ...images], [images]);

  useEffect(() => {
    const handleResize = () => {
      setIsMobile(window.innerWidth <= 768);
    };
    
    const resizeObserver = new ResizeObserver(handleResize);
    resizeObserver.observe(document.body);
    
    return () => resizeObserver.unobserve(document.body);
  }, []);

  useEffect(() => {
    let isMounted = true;
    
    const fetchData = async () => {
      try {
        const response = await apiService.get("galleries");
        if (isMounted) {
          setImages(response.data);
          setLoading(false);
        }
      } catch (err) {
        console.error(err);
        if (isMounted) setError("Failed to fetch images");
      }
    };
    
    fetchData();
    
    return () => {
      isMounted = false;
    };
  }, []);

  if (error) return <p className="text-danger">{error}</p>;
  if (loading) return <div className="placeholder-content" />;

  return (
    <div className={`d-flex ${isMobile ? "flex-column" : "justify-content-end"}`}>
      {/* First Image Scroll */}
      <div className="scroll-container">
        <motion.div
          className="scroll-content"
          animate={{ y: ["0%", "-100%"] }}
          transition={{
            repeat: Infinity,
            duration: 100,
            ease: "linear",
          }}
        >
          {doubledImages.map((img, index) => (
            <LazyImage 
              key={`first-${index}`}
              src={`${storage}/${img}`}
              alt={`Event ${index + 15}`}
            />
          ))}
        </motion.div>
      </div>

      {/* Second Image Scroll */}
      <div className="scroll-container ps-2">
        <motion.div
          className="scroll-content"
          animate={{ y: ["-100%", "0%"] }}
          transition={{
            repeat: Infinity,
            duration: 100,
            ease: "linear",
          }}
        >
          {doubledImages.map((img, index) => (
            <LazyImage 
              key={`second-${index}`}
              src={`${storage}/${img}`}
              alt={`Event ${index + 15}`}
            />
          ))}
        </motion.div>
      </div>
    </div>
  );
};

// Optimized LazyImage component
const LazyImage = React.memo(({ src, alt }) => {
  const [loaded, setLoaded] = useState(false);
  
  return (
    <div className="item">
      {!loaded && <div className="image-placeholder" />}
      <img 
        className={`img-fluid ${loaded ? 'visible' : 'hidden'}`}
        src={src}
        alt={alt}
        loading="lazy"
        onLoad={() => setLoaded(true)}
      />
    </div>
  );
});

export default React.memo(VerticalScroll);