import React, { useRef, useEffect, useState, memo } from "react";
import { Typography, IconButton } from "@mui/material";
import { VolumeOff, VolumeUp } from "@mui/icons-material";
import api from "../../services/apiservices";

const BASE_URL = "https://buildintec.codissia.com/buildintec-backend/public/";

const VideoSection = () => {
  const videoRef = useRef(null);
  const [videos, setVideos] = useState([]);
  const [isMuted, setIsMuted] = useState(true); // initially muted

  useEffect(() => {
    const fetchVideos = async () => {
      try {
        const res = await api.get("/video");
        if (res?.status === "success") {
          setVideos(res.data || []);
        }
      } catch (error) {
        console.error("Error fetching videos:", error);
      }
    };
    fetchVideos();
  }, []);

  useEffect(() => {
    if (videoRef.current) {
      videoRef.current.muted = isMuted;
      videoRef.current.loop = true;
      videoRef.current.play().catch((err) => {
        console.warn("Autoplay blocked:", err);
      });
    }
  }, [videos, isMuted]);

  const toggleMute = () => {
    if (videoRef.current) {
      videoRef.current.muted = !isMuted;
      videoRef.current.play();
      setIsMuted(!isMuted);
    }
  };

  return (
    <div className="section">
      <div>
        {videos.length > 0 ? (
          videos.map((videoItem, index) => (
            <div
              key={videoItem.id || index}
              className="position-relative text-center"
            >
              <video
                ref={index === 0 ? videoRef : null} // autoplay first video
                className="video-section_video img-fluid"
                src={`${BASE_URL}${videoItem.file_path}`}
                autoPlay
                loop
                muted={isMuted}
                playsInline
                style={{ width: "100%", height: "auto", objectFit: "cover" }}
              />
              {index === 0 && (
                <IconButton
                  onClick={toggleMute}
                  sx={{
                    position: "absolute !important",
                    bottom: { xs: 5, sm: 10 }, // smaller top
                    right: { xs: 5, sm: 10 }, // smaller right
                    backgroundColor: "rgba(0,0,0,0.5) !important",
                    color: "#fff !important",
                    "&:hover": {
                      backgroundColor: "rgba(0,0,0,0.7) !important",
                    },
                    "& .MuiSvgIcon-root": {
                      fontSize: { xs: 12, sm: 16 }, // smaller icon size
                    },
                  }}
                >
                  {isMuted ? <VolumeOff /> : <VolumeUp />}
                </IconButton>
              )}
            </div>
          ))
        ) : (
          <Typography variant="body1" align="center">
            Loading video...
          </Typography>
        )}
      </div>
    </div>
  );
};

export default memo(VideoSection);
