import React, { useState, useContext, useEffect } from "react";
import { motion, AnimatePresence } from "framer-motion";
import { Themecontext } from "../../context/Themecontext";
import { storage } from "../../services/apiservices";

const Timeline = () => {
  const { awards = [] } = useContext(Themecontext);
  const [activeTab, setActiveTab] = useState("lifetime");
  const [activeIndex, setActiveIndex] = useState(0);

  // Filter awards - lifetime achievements are those without a category or with null category
  const lifetimeAwards = awards.filter(
    (award) => award.award_category !== "Young Writers Award"
  );

  const youngWriters = awards.filter(
    (award) => award.award_category === "Young Writers Award"
  );

  const currentAwards = activeTab === "lifetime" ? lifetimeAwards : youngWriters;

  const sortedAwards = currentAwards
    ? [...currentAwards].sort((a, b) => b.award_year - a.award_year)
    : [];

  useEffect(() => {
    setActiveIndex(0);
  }, [activeTab]);

  if (awards.length === 0) {
    return <p className="text-center">Loading awards...</p>;
  }

  const handleNext = () => {
    setActiveIndex((prevIndex) =>
      prevIndex === sortedAwards.length - 1 ? 0 : prevIndex + 1
    );
  };

  const handlePrev = () => {
    setActiveIndex((prevIndex) =>
      prevIndex === 0 ? sortedAwards.length - 1 : prevIndex - 1
    );
  };

  return (
    <div className="container">
      <div className="text-center section-title">
        <h4>
          {activeTab === "lifetime"
            ? "வாழ்நாள் சாதனையாளர் விருது"
            : "இளஞர் எழுத்தாளர் விருது"}
        </h4>
        <p>
          {activeTab === "lifetime"
            ? "கோயம்புத்தூர் புத்தகத் திருவிழாவின் சார்பில் ஒவ்வொரு ஆண்டும் தமிழ் இலக்கியத் துறையில் சாதனை படைத்த ஒரு எழுத்து ஆளுமைக்கு வாழ்நாள் சாதனையாளர் விருது வழங்கி சிறப்பிக்கப்பட்டு வருகிறது."
            : "இளஞர் எழுத்தாளர் விருது புதிய தலைமுறை எழுத்தாளர்களை ஊக்குவிக்க வழங்கப்படுகிறது."}
        </p>
      </div>
      {youngWriters.length > 0 && (
        <div className="text-center mb-4">
          <button
            className={`btn ${
              activeTab === "lifetime" ? "btn-awards-active" : "btn-awards"
            } mx-2`}
            onClick={() => setActiveTab("lifetime")}
          >
            வாழ்நாள் சாதனையாளர் விருது
          </button>
          <button
            className={`btn ${
              activeTab === "youngWriters" ? "btn-awards-active" : "btn-awards"
            } mx-2`}
            onClick={() => setActiveTab("youngWriters")}
          >
            இளஞர் எழுத்தாளர் விருது
          </button>
        </div>
      )}

      {/* Timeline Year Tabs */}
      {sortedAwards.length > 0 && (
        <>
          <div className="timeline-tabs text-center mb-4">
            {sortedAwards.map((award, index) => (
              <button
                key={index}
                className={`timeline-tab ${index === activeIndex ? "active" : ""}`}
                onClick={() => setActiveIndex(index)}
              >
                {award.award_year}
              </button>
            ))}
          </div>

          {/* Award Display */}
          <div className="col-12 m-auto">
            <div className="timeline-content text-center mb-5">
              <AnimatePresence mode="wait">
                <motion.div
                  key={activeIndex}
                  initial={{ opacity: 0, y: 50 }}
                  animate={{ opacity: 1, y: 0 }}
                  exit={{ opacity: 0, y: -50 }}
                  transition={{ duration: 0.6, ease: "easeInOut" }}
                  className="row"
                >
                  <div>
                    <motion.img
                      src={
                        sortedAwards[activeIndex]?.image_path
                          ? `${storage}${
                              sortedAwards[activeIndex].image_path.startsWith("/")
                                ? sortedAwards[activeIndex].image_path
                                : `/${sortedAwards[activeIndex].image_path}`
                            }`
                          : "/default-image.jpg"
                      }
                      alt={sortedAwards[activeIndex]?.name || "Award Image"}
                      className="img-fluid mb-3"
                      initial={{ scale: 0.9, opacity: 0 }}
                      animate={{ scale: 1, opacity: 1 }}
                      transition={{ duration: 0.6, ease: "easeOut" }}
                    />
                  </div>
                  <div>
                    {sortedAwards[activeIndex] && (
                      <>
                        <motion.h3 className="timeline-year">
                          {sortedAwards[activeIndex].award_year}
                        </motion.h3>
                        <motion.h4>{sortedAwards[activeIndex].name}</motion.h4>
                        
                      </>
                    )}
                  </div>
                </motion.div>
              </AnimatePresence>

              {/* Navigation Buttons */}
              {sortedAwards.length > 1 && (
                <div className="timeline-nav d-flex justify-content-center mt-3">
                  <button className="btn btn-outline-dark me-2" onClick={handlePrev}>
                    ←
                  </button>
                  <button className="btn btn-outline-dark" onClick={handleNext}>
                    →
                  </button>
                </div>
              )}
            </div>
          </div>
        </>
      )}
    </div>
  );
};

export default React.memo(Timeline);
