import React, { useState, useEffect } from "react";
import {
  Container,
  Grid,
  Card,
  CardMedia,
  Dialog,
  IconButton,
  Box,
} from "@mui/material";
import { motion } from "framer-motion";
import { ArrowBack, ArrowForward, Close, ZoomIn } from "@mui/icons-material";
import apiService, { storage } from "../../services/apiservices";

const EventsGallery = () => {
  const [isOpen, setIsOpen] = useState(false);
  const [photoIndex, setPhotoIndex] = useState(0);
  const [hoverIndex, setHoverIndex] = useState(null);
  const [galleryData, setGalleryData] = useState([]);
  useEffect(() => {
    const fetchData = async () => {
      try {
        console.log("Fetching gallery data...");
        const response = await apiService.get("galleries");
        const data = response.data;
  
        if (Array.isArray(data)) {
          setGalleryData(data);
        } else {
          console.error("Unexpected API response format:", data);
        }
      } catch (error) {
        console.error("Error fetching gallery data:", error);
      }
    };
  
    fetchData();
  }, []);
  

  const handleOpen = (index) => {
    setPhotoIndex(index);
    setIsOpen(true);
  };

  const handleClose = () => setIsOpen(false);
  const prevImage = () =>
    setPhotoIndex((photoIndex + galleryData.length - 1) % galleryData.length);
  console.log(galleryData);
  console.log(photoIndex);
  const nextImage = () => setPhotoIndex((photoIndex + 1) % galleryData.length);

  return (
    <Container className="mt-4 mb-4">
      {/* Image Gallery */}
      <Grid container spacing={3} className="justify-content-center">
        {galleryData.length > 0 ? (
          galleryData.map((src, index) => (
            <Grid item xs={6} sm={6} md={4} key={index}>
              <motion.div
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ duration: 0.5, delay: index * 0.1 }}
                onMouseEnter={() => setHoverIndex(index)}
                onMouseLeave={() => setHoverIndex(null)}
                style={{ position: "relative", cursor: "pointer" }}
              >
                <Card onClick={() => handleOpen(index)}>
                  <CardMedia
                    component="img"
                    height="300"
                    image={`${storage}/${src.image_path}`} 
                    alt={`Gallery ${index + 1}`}
                    loading="lazy"
                  />
                  {hoverIndex === index && (
                    <motion.div
                      initial={{ opacity: 0 }}
                      animate={{ opacity: 1 }}
                      transition={{ duration: 0.3 }}
                      style={{
                        position: "absolute",
                        top: 0,
                        left: 0,
                        width: "100%",
                        height: "100%",
                        background: "rgba(0, 0, 0, 0.4)",
                        backdropFilter: "blur(5px)",
                        display: "flex",
                        justifyContent: "center",
                        alignItems: "center",
                        borderRadius: "4px",
                      }}
                    >
                      <ZoomIn sx={{ fontSize: 50, color: "white" }} />
                    </motion.div>
                  )}
                </Card>
              </motion.div>
            </Grid>
          ))
        ) : (
          <div className="no-testimonials">
            <p>No images available</p>
          </div>
        )}
      </Grid>

      {/* Lightbox Dialog */}
      <Dialog fullScreen open={isOpen} onClose={handleClose}>
        <Box
          display="flex"
          justifyContent="center"
          alignItems="center"
          height="100vh"
          position="relative"
          bgcolor="black"
        >
          <IconButton
            onClick={handleClose}
            sx={{ position: "absolute", top: 20, right: 20, color: "white" }}
          >
            <Close sx={{ fontSize: 30 }} />
          </IconButton>
          <IconButton
            onClick={prevImage}
            sx={{ position: "absolute", left: 20, color: "white" }}
          >
            <ArrowBack sx={{ fontSize: 40 }} />
          </IconButton>
          <motion.img
            key={photoIndex}
            src={`${storage}/${galleryData[photoIndex]?.image_path}`} 
            alt="Lightbox"
            initial={{ opacity: 0, scale: 0.8 }}
            animate={{ opacity: 1, scale: 1 }}
            transition={{ duration: 0.5, ease: "easeOut" }}
            style={{
              maxWidth: "100%",
              maxHeight: "90vh",
              objectFit: "contain",
              borderRadius: "8px",
            }}
          />
          <IconButton
            onClick={nextImage}
            sx={{ position: "absolute", right: 20, color: "white" }}
          >
            <ArrowForward sx={{ fontSize: 40 }} />
          </IconButton>
        </Box>
      </Dialog>
    </Container>
  );
};

export default React.memo(EventsGallery);