import React, { useState } from "react";

const ContactForm = () => {
  const [formData, setFormData] = useState({
    name: "",
    email: "",
    phone: "",
    message: "",
  });

  const [errors, setErrors] = useState({});
  const [successMessage, setSuccessMessage] = useState("");

  // Handle input change
  const handleChange = (e) => {
    setFormData({ ...formData, [e.target.name]: e.target.value });
    setErrors({ ...errors, [e.target.name]: "" });
  };

  // Validate form
  const validateForm = () => {
    let newErrors = {};
    if (!formData.name.trim()) newErrors.name = "Name is required!";
    if (!formData.email.trim()) {
      newErrors.email = "Email is required!";
    } else if (!/\S+@\S+\.\S+/.test(formData.email)) {
      newErrors.email = "Invalid email format!";
    }
    if (!formData.phone.trim()) {
      newErrors.phone = "Phone number is required!";
    } else if (!/^\d{10}$/.test(formData.phone)) {
      newErrors.phone = "Invalid phone number!";
    }
    if (!formData.message.trim()) newErrors.message = "Message is required!";

    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  // Handle form submission
  const handleSubmit = (e) => {
    e.preventDefault();
    if (validateForm()) {
      setSuccessMessage("Your message has been sent successfully!");
      setFormData({ name: "", email: "", phone: "", message: "" });
      setTimeout(() => setSuccessMessage(""), 3000);
    }
  };

  return (
    <div className="contact-form-container">
      <h3 className="form-title">Reach Us</h3>
      {successMessage && <p className="success-message">{successMessage}</p>}

      <form onSubmit={handleSubmit}>
        <div className="form-group">
          <input
            type="text"
            name="name"
            placeholder="Your Name"
            value={formData.name}
            onChange={handleChange}
            className={`form-control ${errors.name ? "input-error" : ""}`}
          />
          {errors.name && <small className="error-text">{errors.name}</small>}
        </div>
        <div className="row">
          <div className="col-6">
            <div className="form-group">
              <input
                type="email"
                name="email"
                placeholder="Your Email"
                value={formData.email}
                onChange={handleChange}
                className={`form-control ${errors.email ? "input-error" : ""}`}
              />
              {errors.email && (
                <small className="error-text">{errors.email}</small>
              )}
            </div>
          </div>
          <div className="col-6">
            <div className="form-group">
              <input
                type="tel"
                name="phone"
                placeholder="Your Phone"
                value={formData.phone}
                onChange={handleChange}
                className={`form-control ${errors.phone ? "input-error" : ""}`}
              />
              {errors.phone && (
                <small className="error-text">{errors.phone}</small>
              )}
            </div>
          </div>
        </div>

        <div className="form-group">
          <textarea
            name="message"
            placeholder="Your Message"
            value={formData.message}
            onChange={handleChange}
            className={`form-control ${errors.message ? "input-error" : ""}`}
          ></textarea>
          {errors.message && (
            <small className="error-text">{errors.message}</small>
          )}
        </div>

        <button type="submit" className="submit-btn">
          Submit
        </button>
      </form>
    </div>
  );
};

export default ContactForm;
