import { useContext, useEffect, useRef } from "react";
import { PropsContext } from "../../../context/PropsContext";
import GalleryCard from "../../ui/cards/GalleryCard";
import PhotoSwipeLightbox from "photoswipe/lightbox";
import "photoswipe/style.css";

export default function Photogallery() {
  const { formattedGallery, galleryPagination, fetchGalleryData } =
    useContext(PropsContext);

  const lightboxRef = useRef(null);

  useEffect(() => {
    if (formattedGallery.length > 0) {
      lightboxRef.current = new PhotoSwipeLightbox({
        gallery: "#gallery-container",
        children: "a[data-pswp-item]",
        pswpModule: () => import("photoswipe"),
        showHideAnimationType: "fade",
        bgOpacity: 0.9,
        errorMsg: "The image cannot be loaded",
      });

      // Custom caption with title
      lightboxRef.current.on("uiRegister", () => {
        lightboxRef.current.pswp.ui.registerElement({
          name: "custom-caption",
          order: 9,
          isButton: false,
          appendTo: "root",
          className: "pswp-custom-caption",
          onInit: (el, pswp) => {
            lightboxRef.current.pswp.on("change", () => {
              const currSlide = pswp.currSlide;
              if (currSlide?.data?.element) {
                const title = currSlide.data.element.querySelector(".project-title h5")?.textContent || "";
                const caption = currSlide.data.element.querySelector("[data-caption]")?.innerHTML || "";
                
                // Create caption HTML with title above description
                el.innerHTML = `
                  <div class="pswp-caption-container">
                    ${title ? `<h3 class="pswp-caption-title">${title}</h3>` : ''}
                    ${caption ? `<div class="pswp-caption-text">${caption}</div>` : ''}
                  </div>
                `;
              }
            });
          },
        });
      });

      lightboxRef.current.init();

      return () => {
        if (lightboxRef.current) {
          lightboxRef.current.destroy();
          lightboxRef.current = null;
        }
      };
    }
  }, [formattedGallery]);

  const handlePageChange = (page) => {
    if (page >= 1 && page <= galleryPagination.last_page) {
      fetchGalleryData(page);
    }
  };

  const isFirstPage = galleryPagination.current_page === 1;
  const isLastPage =
    galleryPagination.current_page === galleryPagination.last_page;

  return (
    <section className="project-sec2 sec-ptb">
      <div className="container">
        <div className="row">
          <div className="offset-lg-3 col-lg-6">
            <div className="sec-content text-center mb-5">
              <h6 className="sec-sub-title">Our Gallery</h6>
              <h2 className="sec-title">Featured Works</h2>
            </div>
          </div>
        </div>

        <div id="gallery-container" className="row">
          {formattedGallery.length > 0 ? (
            formattedGallery.map((item) => (
              <div className="col-md-6 col-lg-4" key={item.id}>
                <a
                  href={item.imageUrl || item.image}
                  data-pswp-item
                  data-pswp-width={item.width || 1200}
                  data-pswp-height={item.height || 800}
                  className="gallery-item"
                  target="_blank"
                  rel="noopener noreferrer"
                >
                  <GalleryCard data={item} />
                  <div className="d-none" data-caption>
                    {item.caption || item.description || ""}
                  </div>
                </a>
              </div>
            ))
          ) : (
            <p className="text-center">No gallery items found.</p>
          )}
        </div>

        <div className="pagination-wrapper text-center mt-4">
          <button
            className="btn btn-outline-secondary mx-1"
            onClick={() => handlePageChange(galleryPagination.current_page - 1)}
            disabled={isFirstPage}
          >
            &laquo;
          </button>

          {[...Array(galleryPagination.last_page)].map((_, i) => (
            <button
              key={i + 1}
              onClick={() => handlePageChange(i + 1)}
              className={`btn btn-sm mx-1 ${
                galleryPagination.current_page === i + 1
                  ? "btn-primary"
                  : "btn-outline-primary"
              }`}
            >
              {i + 1}
            </button>
          ))}

          <button
            className="btn btn-outline-secondary mx-1"
            onClick={() => handlePageChange(galleryPagination.current_page + 1)}
            disabled={isLastPage}
          >
            &raquo;
          </button>
        </div>
      </div>

    </section>
  );
}