import React, { useRef, useEffect, useState } from "react";
import { gsap } from "gsap";

const ImageHoverGallery = () => {
  const imageRefs = useRef([]);
  const wrapRef = useRef(null);
  const [isMobile, setIsMobile] = useState(false);
  const duration = 0.6;

  // Responsive width values
  const defaultWidth = isMobile ? "100%" : "20%";
  const expandedWidth = isMobile ? "100%" : "40%";
  const contractedWidth = isMobile ? "100%" : "15%";

  const images = [
    {
      src: "/images/home/platfrom-1.png",
      title: "Innovation",
      description:
        "Fostering creative thinking and novel solutions to drive progress",
    },
    {
      src: "/images/home/platfrom-2.png",
      title: "Indigenization",
      description: "Adapting global knowledge to local contexts and needs",
    },
    {
      src: "/images/home/platfrom-3.png",
      title: "Incubation",
      description: "Nurturing ideas and startups with resources and mentorship",
    },
    {
      src: "/images/home/platfrom-4.png",
      title: "Integration",
      description: "Combining diverse elements into a unified system",
    },
    {
      src: "/images/home/platfrom-5.png",
      title: "Implementation",
      description: "Turning ideas into practical solutions and products",
    },
  ];

  useEffect(() => {
    const checkMobile = () => {
      setIsMobile(window.innerWidth <= 992);
    };

    // Initial check
    checkMobile();

    // Add resize listener
    window.addEventListener("resize", checkMobile);

    // Cleanup
    return () => window.removeEventListener("resize", checkMobile);
  }, []);

useEffect(() => {
  if (isMobile) {
    // Mobile layout - stack vertically
    gsap.set(imageRefs.current, {
      width: "100%",
      height: "auto",
      minHeight: "200px",
      filter: "grayscale(0%)",
    });
  } else {
    // Desktop layout - horizontal with hover effects
    gsap.set(imageRefs.current, {
      width: defaultWidth,
      height: "100%",
      filter: "grayscale(100%)",
    });

    const listeners = [];

    const handleMouseEnter = (index) => {
      imageRefs.current.forEach((img, i) => {
        gsap.to(img, {
          duration,
          width: i === index ? expandedWidth : contractedWidth,
          filter: i === index ? "grayscale(0%)" : "grayscale(100%)",
        });
      });
    };

    const handleMouseLeave = () => {
      gsap.to(imageRefs.current, {
        duration,
        width: defaultWidth,
        filter: "grayscale(100%)",
      });
    };

    // Attach listeners
    imageRefs.current.forEach((img, index) => {
      if (img) {
        const enterHandler = () => handleMouseEnter(index);
        listeners.push({ img, enterHandler });
        img.addEventListener("mouseenter", enterHandler);
      }
    });

    wrapRef.current?.addEventListener("mouseleave", handleMouseLeave);

    // Cleanup
    return () => {
      listeners.forEach(({ img, enterHandler }) => {
        img.removeEventListener("mouseenter", enterHandler);
      });
      wrapRef.current?.removeEventListener("mouseleave", handleMouseLeave);
    };
  }
}, [isMobile]);

  return (
    <>
      <section className="image-hover-gallery d-none d-lg-block">
        <div className="container">
          <div class="sec-title text-center news-letter mb-2">
            <h2>CDIIC – Platform</h2>
          </div>
        </div>
        <div>
          <div ref={wrapRef} className={`options-sec `}>
            {images.map((image, index) => (
              <div
                key={index}
                ref={(el) => (imageRefs.current[index] = el)}
                className={`img-${index + 1} photo`}
              >
                <img src={image.src} alt={image.title} />
                <div className="image-label">{image.title}</div>
                <div className="image-details">
                  <h3>{image.title}</h3>
                  <p>{image.description}</p>
                </div>
              </div>
            ))}
          </div>
        </div>

        {/* --------- Mobile ---------- */}
      </section>
      <section className="d-block d-lg-none image-hover-gallery-main">
        <div className="container">
          <div class="sec-title text-center news-letter mb-3">
            <h2>CDIIC – Platform</h2>
          </div>
          <div className="row d-flex justify-content-center align-items-center">
            {images.map((image, index) => (
              <div className="col-lg-4 col-md-6 col-sm-12 d-flex justify-content-center align-items-center mb-3">
                <div key={index} className="events-section-mobile">
                  <img src={image.src} alt={image.title} />

                  <div className="events-section-mobile-main">
                    <h3>{image.title}</h3>
                    <p>{image.description}</p>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      </section>
    </>
  );
};

export default ImageHoverGallery;
