import React, { useEffect, useRef, useContext, useState } from "react";
import { gsap } from "gsap";
import { ScrollTrigger } from "gsap/ScrollTrigger";
import { PropsContext } from "../../../context/PropsContext";
import { FaFilePdf } from "react-icons/fa";

gsap.registerPlugin(ScrollTrigger);

const TimelineCard = () => {
  const cardsRef = useRef([]);
  const { 
    formattedTenders, 
    tendersPagination, 
    fetchTendersData 
  } = useContext(PropsContext);
  const [filter, setFilter] = useState("all"); // 'all', 'active', 'archived'
  const [filteredTenders, setFilteredTenders] = useState([]);

  // Apply filter when tenders data or filter changes
  useEffect(() => {
    if (!formattedTenders) return;

    const filtered = formattedTenders.filter((tender) => {
      if (filter === "all") return true;
      return tender.status.toLowerCase() === filter.toLowerCase();
    });
    setFilteredTenders(filtered);
  }, [formattedTenders, filter]);

  const getBgClass = (status) => {
    switch (status.toLowerCase()) {
      case "active":
        return "bg-light-green";
      case "archived":
        return "bg-light-pink";
      default:
        return "bg-light";
    }
  };

  const getStatusBadgeClass = (status) => {
    switch (status.toLowerCase()) {
      case "active":
        return "bg-secondary";
      case "archived":
        return "bg-secondary";
      default:
        return "bg-secondary";
    }
  };

  useEffect(() => {
    cardsRef.current = cardsRef.current.slice(0, filteredTenders.length);

    cardsRef.current.forEach((card) => {
      gsap.fromTo(
        card,
        { opacity: 0, y: 40 },
        {
          opacity: 1,
          y: 0,
          duration: 1.2,
          ease: "power3.out",
          scrollTrigger: {
            trigger: card,
            start: "top 85%",
          },
        }
      );
    });
  }, [filteredTenders]);

  const handlePageChange = (newPage) => {
    if (newPage < 1 || newPage > tendersPagination.last_page) return;
    fetchTendersData(newPage);
  };

  if (!formattedTenders || formattedTenders.length === 0) {
    return (
      <div className="container py-5 text-center">
        <p>Loading.....</p>
      </div>
    );
  }

  return (
    <section className="tenders-main">
      <div className="container-fluid py-5">
        <div className="row">
          {/* Sidebar for filtering */}
          <div className="col-lg-3 mb-4 mb-lg-0">
            <div className=" sticky-top" style={{ top: "20px" }}>
              <div className="sidebar-card">
                <div className="d-flex flex-column gap-3">
                  <button
                    className={`filter-btn ${
                      filter === "all" ? "active all" : ""
                    }`}
                    onClick={() => setFilter("all")}
                  >
                    <span className="filter-icon">
                      <svg
                        width="20"
                        height="20"
                        viewBox="0 0 24 24"
                        fill="none"
                      >
                        <path
                          d="M4 6H20M4 12H20M4 18H20"
                          stroke="currentColor"
                          strokeWidth="2"
                          strokeLinecap="round"
                        />
                      </svg>
                    </span>
                    <span className="filter-text">All Tenders</span>
                    <span className="filter-count">
                      {formattedTenders.length}
                    </span>
                  </button>

                  <button
                    className={`filter-btn ${
                      filter === "active" ? "active active" : ""
                    }`}
                    onClick={() => setFilter("active")}
                  >
                    <span className="filter-icon">
                      <svg
                        width="20"
                        height="20"
                        viewBox="0 0 24 24"
                        fill="none"
                      >
                        <path
                          d="M5 13L9 17L19 7"
                          stroke="currentColor"
                          strokeWidth="2"
                          strokeLinecap="round"
                          strokeLinejoin="round"
                        />
                      </svg>
                    </span>
                    <span className="filter-text">Active Tenders</span>
                    <span className="filter-count">
                      {
                        formattedTenders.filter(
                          (t) => t.status.toLowerCase() === "active"
                        ).length
                      }
                    </span>
                  </button>

                  <button
                    className={`filter-btn ${
                      filter === "archived" ? "active archived" : ""
                    }`}
                    onClick={() => setFilter("archived")}
                  >
                    <span className="filter-icon">
                      <svg
                        width="20"
                        height="20"
                        viewBox="0 0 24 24"
                        fill="none"
                      >
                        <path
                          d="M19 7L5 7M14 12H5M12 17H5"
                          stroke="currentColor"
                          strokeWidth="2"
                          strokeLinecap="round"
                        />
                      </svg>
                    </span>
                    <span className="filter-text">Archived Tenders</span>
                    <span className="filter-count">
                      {
                        formattedTenders.filter(
                          (t) => t.status.toLowerCase() === "archived"
                        ).length
                      }
                    </span>
                  </button>
                </div>
              </div>
            </div>
          </div>

          {/* Main content area */}
          <div className="col-lg-9">
            {filteredTenders.length === 0 ? (
              <div className="text-center py-5">
                <h5 className="text-white">No tenders match your filter</h5>
                <p className="text-white">
                  Try selecting a different filter option
                </p>
              </div>
            ) : (
              filteredTenders.map((tender, index) => (
                <div
                  className="d-flex position-relative timeline-card mb-5"
                  key={tender.id}
                  ref={(el) => (cardsRef.current[index] = el)}
                >
                  <div className="timeline-line">
                    <div className="circle-animate" />
                  </div>
                  <div className={`card flex-fill p-4 shadow custom-card `}>
                    <div>
                      <div>
                        <h5 className="fw-bold mb-2">{tender.title}</h5>
                        <p className="text-muted mb-2">{tender.details}</p>
                        <div className="d-flex flex-wrap gap-3">
                          {tender.credits.map((credit, i) => (
                            <small key={i} className={credit.className}>
                              {credit.icon} {credit.text}
                            </small>
                          ))}
                        </div>
                        {tender.pdfUrl && (
                          <div className="d-flex ">
                            <a
                              href={tender.pdfUrl}
                              target="_blank"
                              rel="noopener noreferrer"
                              className="btn btn-outline-dark btn-sm mt-3 rounded px-3 p-2 d-flex align-items-center gap-1"
                            >
                              <FaFilePdf /> Download Tender
                            </a>
                          </div>
                        )}
                      </div>
                      <div className="text-end active-status">
                        <span
                          className={`badge  px-3 py-2 rounded-pill ${getStatusBadgeClass(
                            tender.status
                          )}`}
                        >
                          {tender.status}
                        </span>
                        <br />
                      </div>
                    </div>
                  </div>
                </div>
              ))
            )}
          </div>
        </div>
        {/* Pagination */}
        <div className="pagination-wrapper text-center mt-4">
          <button
            className="btn btn-outline-secondary mx-1"
            onClick={() => handlePageChange(tendersPagination.current_page - 1)}
            disabled={tendersPagination.current_page === 1}
          >
            &laquo;
          </button>

          {[...Array(tendersPagination.last_page)].map((_, i) => (
            <button
              key={i + 1}
              onClick={() => handlePageChange(i + 1)}
              className={`btn btn-sm mx-1 ${
                tendersPagination.current_page === i + 1
                  ? "btn-primary"
                  : "btn-outline-primary"
              }`}
            >
              {i + 1}
            </button>
          ))}

          <button
            className="btn btn-outline-secondary mx-1"
            onClick={() => handlePageChange(tendersPagination.current_page + 1)}
            disabled={tendersPagination.current_page === tendersPagination.last_page}
          >
            &raquo;
          </button>
        </div>
      </div>
    </section>
  );
};

export default TimelineCard;