import { useRef, useEffect } from "react";
import { useDraggable } from "react-use-draggable-scroll";
import { LuArrowBigRightDash } from "react-icons/lu";
import { LuArrowBigLeftDash } from "react-icons/lu";

export default function VintageTimeline() {
  const timelineRef = useRef();
  const { events } = useDraggable(timelineRef);

  useEffect(() => {
    const cards = document.querySelectorAll(".timeline-card");
    const timeline = timelineRef.current;

    const handleScroll = () => {
      const timelineRect = timeline.getBoundingClientRect();
      const timelineCenter = timelineRect.left + timelineRect.width / 2;

      cards.forEach((card) => {
        const cardRect = card.getBoundingClientRect();
        const cardCenter = cardRect.left + cardRect.width / 2;
        const distanceFromCenter =
          (cardCenter - timelineCenter) / timelineRect.width;

        // Parallax effect
        const parallaxOffset = distanceFromCenter * 100;
        card.style.transform = `translateX(${parallaxOffset * 0.5}px)`;

        // Scale effect for center card
        const scale = 1 - Math.abs(distanceFromCenter) * 0.1;
        card.style.transform += ` scale(${scale})`;

        // Opacity effect
        const opacity = 0.8 + (1 - Math.abs(distanceFromCenter)) * 0.2;
        card.style.opacity = opacity;
      });
    };

    timeline.addEventListener("scroll", handleScroll);
    window.addEventListener("resize", handleScroll);
    handleScroll(); // Initialize

    return () => {
      timeline.removeEventListener("scroll", handleScroll);
      window.removeEventListener("resize", handleScroll);
    };
  }, []);

  // Timeline data
  const timelineData = [
    {
      year: "1969",
      title: "Foundation",
      content:
        "The Coimbatore District Small Scale Industries Association (CODISSIA) was formed with 40 members and was registered under the Tamilnadu Societies Registration Act vide registration number 80/1969 dated 17-11-69",
      icon: "🏛️",
    },
    {
      year: "1970",
      title: "First Achievements",
      content:
        "At the instance of CODISSIA, SIDCO Branch Office was opened. Kurichi Industrial Estate was formed in Coimbatore.",
      icon: "🏭",
    },
    {
      year: "1972",
      title: "International Reach",
      content:
        "Members of CODISSIA participated in ASIA-71 Trade Fair at New Delhi. CODISSIA organised one Vendor Exhibition in Association with ICF, Chennai.",
      icon: "🌍",
    },
    {
      year: "1973",
      title: "Expansion",
      content:
        "A Government of India committee on Electric Motors and Pumps visited Coimbatore and suggested industrial diversification. CODISSIA launched its *Bulletin* journal, and members undertook study tours to Japan and South Asian countries.",
      icon: "📈",
    },
    {
      year: "1974",
      title: "Exhibitions",
      content:
        "An exhibition on Railway and Telecommunication parts was held by CODISSIA, attended by Dr. P.C. Alexander, then Development Commissioner (SSI), Government of India. The Coimbatore Industries Export House was formed, and a study tour to Bangalore was organized for members.",
      icon: "🖼️",
    },
    {
      year: "1975",
      title: "Seminar & Study Tour",
      content:
        "An All India Seminar on 'Rural Industrialisation' was organised. A study tour of members to the Industrial Exhibition at Bombay was conducted.",
      icon: "📚",
    },
    {
      year: "1977",
      title: "Key Visit & Exhibition",
      content:
        "Mr. I.C. Puri, Development Commissioner (SSI), Union Ministry of Industry, New Delhi visited CODISSIA. NSIC Ltd. organised a consumer products exhibition at Coimbatore.",
      icon: "🧾",
    },
    {
      year: "1978",
      title: "Raw Material Initiatives",
      content:
        "The rake load scheme of pig iron was implemented in Coimbatore with CODISSIA’s efforts. CODISSIA Raw Material Society (CRMS) was formed.",
      icon: "⚙️",
    },
    {
      year: "1979",
      title: "Exhibition Initiatives",
      content:
        "CODISSIA organized a display exhibition and a School Laboratory Equipment Exhibition.",
      icon: "🧪",
    },
    {
      year: "1980",
      title: "Logistics Support",
      content:
        "CRMS took over the responsibility of coke transportation to foundries as SIDCO’s agent, solving logistics bottlenecks.",
      icon: "🚚",
    },
    {
      year: "1981",
      title: "Excise Duty Relief",
      content:
        "Through CODISSIA’s efforts, Central Excise duty on electric motor parts was waived, benefitting over 100 small scale manufacturers.",
      icon: "💸",
    },
    {
      year: "1982",
      title: "ENDEX '82",
      content:
        "CODISSIA conducted 'ENDEX 82', a 10-day Entrepreneurship Development Exhibition and Seminar. Union Minister Shri P.A. Sangma attended the valedictory function.",
      icon: "🎪",
    },
    {
      year: "1983",
      title: "Power Supply Reform",
      content:
        "CODISSIA’s efforts led to the conversion of rural feeders into industrial feeders, ensuring 24-hour power for thousands of industries.",
      icon: "🔌",
    },
    {
      year: "1984",
      title: "Own Office Land & Coke Card Reform",
      content:
        "CODISSIA purchased 20 cents of land for its office. Validity of coke cards extended from 1 to 4 years due to CODISSIA's advocacy.",
      icon: "🏢",
    },
    {
      year: "1985",
      title: "Women & Electronics Focus",
      content:
        "Launched entrepreneurial development programs for women. Organised a national seminar and exhibition 'Open ’85' on electronic industry opportunities.",
      icon: "👩‍💼",
    },
    {
      year: "1986",
      title: "IMTEX Tour & Protest",
      content:
        "Members toured IMTEX ’86 in Bombay. A protest Bundh against Central Excise policies was held, including a procession to the Excise office.",
      icon: "✊",
    },
    {
      year: "1987",
      title: "Standardisation & SiTARC Launch",
      content:
        "Initiated Company Standardisation with BIS & SEI. Co-promoted SiTARC, a testing and research centre with IDBI support.",
      icon: "🧬",
    },
    {
      year: "1988",
      title: "INTEC ’88 – A Landmark Fair",
      content:
        "Organised INTEC ’88 — the first industrial trade fair in PSG Grounds. Hosted over 310 stalls, generating Rs.2 crore business. Tamil Nadu Governor Dr. P.C. Alexander visited and addressed members.",
      icon: "🏟️",
    },
    {
      year: "1989",
      title: "Training & Awards",
      content:
        "Organised multiple seminars and trainings on pumps, DC motors, textile machinery, food tech, and export marketing. Launched CODISSIA Awards.",
      icon: "🎓",
    },
    {
      year: "1990",
      title: "Tech Exchange",
      content:
        "Held a foreign products catalogue display. UNIDO expert Mr. M.L. Tartavel visited CODISSIA to study pump industry technologies.",
      icon: "🌐",
    },
    {
      year: "1991",
      title: "INTEC 1991 & B2B Initiatives",
      content:
        "Organized INTEC 1991 — The 4th Industrial Trade Fair. Introduced the concept of Buyers-Sellers Meets to promote B2B industrial interaction in Coimbatore.",
      icon: "🤝",
    },
    {
      year: "1992",
      title: "International Recognition",
      content:
        "CODISSIA was acknowledged for its contributions to the small-scale sector. Hosted a delegation from Sweden’s Small Industries Association to foster global industry exchange.",
      icon: "🌍",
    },
    {
      year: "1993",
      title: "Launch of Agri Intex",
      content:
        "Launched Agri Intex — Tamil Nadu’s premier agriculture and rural technology exhibition. Began digitizing membership records — a tech milestone at the time.",
      icon: "🌾",
    },
    {
      year: "1994",
      title: "Future Vision & Land Acquisition",
      content:
        "Initiated land acquisition for the CODISSIA Trade Fair Complex (CTFC). Successfully hosted INTEC 1994 with increased international participation.",
      icon: "🏗️",
    },
    {
      year: "1995",
      title: "International Partnerships",
      content:
        "Organized a business meet with Taiwan Chamber of Commerce. Partnered with SIDBI to roll out industry-focused loan schemes for MSMEs.",
      icon: "🌐",
    },
    {
      year: "1996",
      title: "CTFC Construction Begins",
      content:
        "Commenced construction of the CODISSIA Trade Fair Complex — a landmark project to empower industries with world-class exhibition infrastructure.",
      icon: "🚧",
    },
    {
      year: "1997",
      title: "INTEC 1997 & Rising Recognition",
      content:
        "Held INTEC 1997 with record participation. CODISSIA's model was appreciated by other state industry associations for replication.",
      icon: "📈",
    },
    {
      year: "1998",
      title: "Infrastructure & Collaboration",
      content:
        "Strengthened ties with national industry bodies. Work progressed rapidly on CTFC’s first phase to host large-scale trade fairs.",
      icon: "🏢",
    },
    {
      year: "1999",
      title: "Digital Leap & Strategic Alliances",
      content:
        "Launched a digital database for members. Collaborated with CII and FICCI to boost regional industrial development.",
      icon: "💻",
    },
    {
      year: "2000",
      title: "Millennium Milestone – CTFC Launch",
      content:
        "Inaugurated the CODISSIA Trade Fair Complex (CTFC) — South India’s largest exhibition venue, marking a new era in industrial promotion.",
      icon: "🎉",
    },
    {
      year: "2001",
      title: "Events & Exhibitions",
      content:
        "AGRI INTEX 2001 was held from 1–7 Aug with ministers and dignitaries participating. Interaction meets with IAS officials, training programs, business workshops, and visits by 28 foreign trainees took place. CODISSIA also conducted Udyog Expo 2001, hosted international delegates, and released a publication on WTO Agreement on Agriculture.",
      icon: "📘",
    },
    {
      year: "2002",
      title: "Events & Collaborations",
      content:
        "CODISSIA organized major expos like AGRI INTEX 2002, Build Expo 2003, and EURO INDIA 2003; signed MoUs with top institutions. It hosted international delegations from Sri Lanka, South Korea, Ethiopia, and Canada, fostering global ties. Dignitaries including ministers, governors, and industry leaders visited for collaborations, discussions, and recognitions.",
      icon: "🌐",
    },
    {
      year: "2003",
      title: "Events & Collaborations",
      content:
        "CODISSIA signed MoUs with TUV Rheinland and Kongu Engineering College. It organized major expos like Transautotec Expo and AGRI INTEX 2003, inaugurated by Tamil Nadu ministers and the Governor. International collaborations included meetings with German textile experts, a Korea show, and delegations from Russia, Afghanistan, and 15 other countries. Seminars, training, and government programs like NAMT were also conducted.",
      icon: "📘",
    },
    {
      year: "2004",
      title: "Strategic Ties & Industry Expos",
      content:
        "CODISSIA strengthened global links through discussions with Silk Air, Sri Lankan and Czech delegations, and US & Korean trade reps. AGRI INTEX 2004, Build Expo 2005, and SUBCON 2005 were successfully held. Key collaborations with NSIC, SIDBI, and various government dignitaries emphasized small-scale industry growth and knowledge exchange.",
      icon: "📘",
    },
    {
      year: "2005",
      title: "Growth, Outreach & Infrastructure",
      content:
        "CODISSIA hosted Build Expo, SUBCON, INTEC, and AGRI INTEX 2005; welcomed delegates from Czech Republic and the US. It launched new facilities including the Library Hall and Conference Hall, and held key events like the Loan Mela, Buyer-Seller Meet, and IPROS Catalogue Show. Relief efforts for Tsunami victims were also organized.",
      icon: "📘",
    },

    {
      year: "2007",
      title: "International Engagements & Infrastructure Developments",
      content:
        "CODISSIA hosted various international delegations, organized seminars and meetings on industrial development, credit schemes, and MSME policies. The Trade Fair Complex underwent major expansions, and notable events included the inauguration of AGRI INTEX 2007, CODISSIA’s recognition in the Limca Book of Records, and the establishment of MOUs with institutes like Vivekanandha and CIET.",
      icon: "🌍",
    },

    {
      year: "2008",
      title: "Global Recognition & Industry Outreach",
      content:
        "CODISSIA continued its global outreach with international delegations from Africa, Germany, and Japan. INTEC 2008 and other major exhibitions like SME Loan Fair and AGRI INTEX were successfully hosted. Additionally, CODISSIA launched its website, marking a digital milestone. The Parliamentary Standing Committee on Commerce also visited Coimbatore for infrastructure development discussions.",
      icon: "🌐",
    },
    {
      year: "2009",
      title: "International Collaborations & Industry Events",
      content:
        "CODISSIA saw continued global collaborations with delegations from Esslingen, Germany, Singapore, and international exhibitions like Build Intec and Intec 2009. The focus was on promoting manufacturing, innovation, and export opportunities, with discussions on the Vision 2020 Global Manufacturing Cluster and industrial growth strategies.",
      icon: "🔧",
    },
    {
      year: "2010",
      title: "Expansion & Networking",
      content:
        "In 2010, CODISSIA organized key exhibitions like Build Intec and Intec, and facilitated interactions with industry leaders such as the Tamil Nadu Chief Minister, and delegates from Germany and Singapore. The year also featured significant events like the World Classical Tamil Conference and interactions with UNIDO representatives, highlighting CODISSIA’s role in supporting industrial development.",
      icon: "🏗️",
    },
    {
      year: "2011",
      title: "Land Expansion & Industrial Advocacy",
      content:
        "CODISSIA began the process of land selection for an industrial estate. The year also saw protests for power distribution reforms and key industrial events like Subcon 2011. Noteworthy interactions included student meetings with Dr. A.P.J. Abdul Kalam and various government and industry officials, continuing CODISSIA's advocacy for MSME development.",
      icon: "💼",
    },
    {
      year: "2012",
      title: "Events & Industry Advancements",
      content:
        "CODISSIA hosted Intec 2012 and Build Intec 2012 exhibitions and organized a Job Fair to support employment. A key event in 2012 was the welcoming of Dr. Farooq Abdullah, a central minister, alongside other notable events, such as the awarding of the Iconic Brand of Coimbatore and B2B meetings with Taiwanese delegates.",
      icon: "📅",
    },
    {
      year: "2013",
      title: "Energy Advocacy & International Collaborations",
      content:
        "CODISSIA took a strong stand for energy reforms during a one-day bandh. Key exhibitions included Elektrotec and Renergy 2013. The year also featured international engagement with countries like Japan and Australia, alongside awards and recognitions for CODISSIA’s efforts in MSME training and development.",
      icon: "⚡",
    },
    {
      year: "2014",
      title: "Delegations & Industry Partnerships",
      content:
        "CODISSIA hosted delegations from the Jaipur Foundry Cluster and held discussions on the upcoming Select USA Summit. The year also involved various meetings with MSME officials, contributing to further expansion of industry partnerships.",
      icon: "🤝",
    },
    {
      year: "2015",
      title: "Continued Growth & Initiatives",
      content:
        "2015 focused on continued growth with contributions from CODISSIA’s team. Specific updates for the year were added based on the team’s suggestions, enhancing industrial partnerships and outreach.",
      icon: "📈",
    },
    {
      year: "2016",
      title: "Further Expansion & Team Contributions",
      content:
        "Further updates to CODISSIA’s initiatives were made in line with team suggestions, continuing the momentum of growth and engagement.",
      icon: "🏢",
    },
    {
      year: "2017",
      title: "Strategic Development & Networking",
      content:
        "Similar to 2016, this year involved more updates and initiatives driven by team suggestions. CODISSIA continued its strategic development efforts.",
      icon: "🚀",
    },
    {
      year: "2018",
      title: "National Outreach & Key Initiatives",
      content:
        "Notable events in 2018 included the President’s Delhi Visit, discussions on MSME policies, outreach programs, and industrial collaboration initiatives. CODISSIA also hosted various meetings with government officials, industry leaders, and held multiple sessions on business growth and ease of doing business.",
      icon: "🇮🇳",
    },
    {
      year: "2019",
      title: "Global Engagement & Industry Growth",
      content:
        "CODISSIA engaged with international delegates from Bangladesh, signed MOUs with Thailand’s Indo-Thai Chamber, and organized various outreach programs and meetings. Notable events included the launch of the Tamil Nadu Defence Industrial Corridor, workshops, and discussions on MSME growth, ease of business, and soft funding for innovation.",
      icon: "🌎",
    },
  ];

  return (
    <section className="milestone-info-section">
      <div className="container my-5">
  

        {/* Timeline with connector line */}
        <div className="position-relative mb-4">
          {/* Connector line */}
          <div className="timeline-connector"></div>

          {/* Scrollable timeline */}
          <div
            ref={timelineRef}
            {...events}
            className="timeline-container d-flex flex-nowrap overflow-auto pt-5"
            style={{
              cursor: "grab",
              scrollbarWidth: "none",
              msOverflowStyle: "none",
              minHeight: "200px",
              position: "relative",
              zIndex: 2,
            }}
          >
            {/* Timeline items */}
            {timelineData.map((item, index) => (
              <div key={index} className="position-relative">
                {/* Year indicator on connector */}
                <div
                  className="position-absolute top-0 start-0 translate-middle-y"
                  style={{
                    left: `${(index / (timelineData.length - 1)) * 80 + 10}%`,
                    zIndex: 3,
                    marginTop: "-10px",
                  }}
                >
                  <div className="year-indicator text-white rounded-pill px-3 py-1 fw-bold shadow-sm">
                    {item.year}
                  </div>
                </div>

                {/* Card */}
                <div
                  className="timeline-card flex-shrink-0 mx-3"
                  style={{
                    width: "320px",
                    transition: "all 0.4s ease-out",
                  }}
                >
                  <div className="card h-100 border-0 shadow-lg overflow-hidden">
                    {/* Card header with icon */}
                    <div className="card-header d-flex align-items-center">
                      <div
                        className="timeline-icon bg-white rounded-circle d-flex align-items-center justify-content-center me-3"
                        style={{ width: "40px", height: "40px" }}
                      >
                        <span style={{ fontSize: "20px" }}>{item.icon}</span>
                      </div>
                      <h5 className="card-title text-white mb-0">
                        {item.title}
                      </h5>
                    </div>

                    {/* Card body with gradient background */}
                    <div className="card-body position-relative p-4">
                      <div
                        className="position-absolute top-0 start-0 w-100 h-100 timeline-card-bg"
                        style={{
                          zIndex: -1,
                        }}
                      ></div>
                      <p
                        className="card-text mb-0"
                        style={{
                          lineHeight: "1.7",
                          fontSize: "14px",
                        }}
                      >
                        {item.content}
                      </p>
                    </div>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>

        {/* Navigation controls */}
        <div className="d-flex justify-content-center mt-4">
          <button
            className="btn-milestone-prev d-flex align-items-center"
            onClick={() => {
              timelineRef.current.scrollBy({ left: -400, behavior: "smooth" });
            }}
          >
            <LuArrowBigLeftDash />
          </button>
          <button
            className="btn-milestone-center fw-bold d-flex align-items-center"
            onClick={() => {
              const center =
                timelineRef.current.scrollWidth / 2 -
                timelineRef.current.clientWidth / 2;
              timelineRef.current.scrollTo({
                left: center,
                behavior: "smooth",
              });
            }}
          >
            Center
          </button>
          <button
            className="btn-milestone-next d-flex align-items-center"
            onClick={() => {
              timelineRef.current.scrollBy({ left: 400, behavior: "smooth" });
            }}
          >
            <LuArrowBigRightDash />
          </button>
        </div>
      </div>
    </section>
  );
}
