import React, { useState } from "react";
import data from "../../utils/utils";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import {
  faEnvelope,
  faMapMarkerAlt,
  faPhone,
} from "@fortawesome/free-solid-svg-icons";

const Officebearersinfo = () => {
  const officebearersData = data();
  const [showModal, setShowModal] = useState(false);
  const [selectedMember, setSelectedMember] = useState(null);

  if (!officebearersData?.officebearers?.length) {
    return <p className="text-center mt-4">No office bearers found.</p>;
  }

  const [firstMember, ...remainingMembers] = officebearersData.officebearers;

  const handleMemberClick = (member) => {
    if (window.innerWidth <= 992) {
      setSelectedMember(member);
      setShowModal(true);
    }
  };

  // Function to chunk members with special handling for last two rows
  const chunkMembers = (members) => {
    const chunks = [];
    const totalMembers = members.length;
    let i = 0;
    
    // Calculate how many 3-member rows we can have before the last two 4-member rows
    const regularRows = Math.max(0, Math.ceil((totalMembers - 8) / 3));
    
    // Create regular rows (3 members each)
    while (i < regularRows * 3 && i < totalMembers) {
      chunks.push(members.slice(i, i + 3));
      i += 3;
    }
    
    // Create last two rows (4 members each)
    while (i < totalMembers) {
      const remaining = totalMembers - i;
      const size = remaining >= 4 || chunks.length >= regularRows + 1 ? 4 : remaining;
      chunks.push(members.slice(i, i + size));
      i += size;
    }
    
    return chunks;
  };

  const memberChunks = chunkMembers(remainingMembers);

  return (
    <section className="office-bearers-section">
      <div className="container mt-4">
        {/* Top centered single member */}
        <div className="row justify-content-center mb-5">
          <div className="col-md-4">
            <MemberCard member={firstMember} onClick={() => handleMemberClick(firstMember)} />
          </div>
        </div>

        {/* Member rows */}
        {memberChunks.map((chunk, chunkIndex) => (
          <div key={`row-${chunkIndex}`} className="row justify-content-center mb-4">
            {chunk.map((member, index) => (
              <div
                key={`member-${chunkIndex}-${index}`}
                // Use col-md-3 for 4 items in a row, col-md-4 for 3 items
                className={`col-sm-6 col-md-${chunk.length === 4 ? '3' : '4'}`}
              >
                <MemberCard member={member} onClick={() => handleMemberClick(member)} />
              </div>
            ))}
          </div>
        ))}
      </div>

      {/* Modal for small screens */}
      {showModal && selectedMember && (
        <div className="custom-office-modal">
          <div className="custom-modal-inner">
            <button className="custom-close-btn" onClick={() => setShowModal(false)}>
              &times;
            </button>
            <h5 className="mb-3">
              {selectedMember.name} - {selectedMember.designation}
            </h5>
            <table className="table table-bordered">
              <tbody>
                <tr>
                  <th>Company</th>
                  <td>{selectedMember.company_name}</td>
                </tr>
                {selectedMember.email && (
                  <tr>
                    <th>Email</th>
                    <td>
                      <FontAwesomeIcon icon={faEnvelope} className="me-2" />
                      <a href={`mailto:${selectedMember.email}`}>{selectedMember.email}</a>
                    </td>
                  </tr>
                )}
                {selectedMember.contact_number && (
                  <tr>
                    <th>Phone</th>
                    <td>
                      <FontAwesomeIcon icon={faPhone} className="me-2" />
                      {selectedMember.contact_number}
                    </td>
                  </tr>
                )}
                <tr>
                  <th>Address</th>
                  <td>
                    <FontAwesomeIcon icon={faMapMarkerAlt} className="me-2" />
                    {selectedMember.address}
                  </td>
                </tr>
              </tbody>
            </table>
          </div>
        </div>
      )}
    </section>
  );
};

const MemberCard = ({ member, onClick }) => (
  <div className="office-bearers mb-4" onClick={onClick}>
    <img src={member.image} alt={member.name} className="card-img-top" />
    <div className="office-bearers-info">
      <h5 className="card-title">{member.name}</h5>
      <strong>{member.designation}</strong>
    </div>
    <div className="office-bearers-details-main">
      <div className="office-bearers-details">
        <h4>{member.company_name}</h4>
        {member.email && (
          <p>
            <FontAwesomeIcon icon={faEnvelope} className="me-2" />
            Email: <a href={`mailto:${member.email}`}>{member.email}</a>
          </p>
        )}
        {member.contact_number && (
          <p>
            <FontAwesomeIcon icon={faPhone} className="me-2" />
            Phone: {member.contact_number}
          </p>
        )}
        <p>
          <FontAwesomeIcon icon={faMapMarkerAlt} className="me-2" />
          Address: {member.address}
        </p>
      </div>
    </div>
  </div>
);

export default  React.memo(Officebearersinfo);