import React, { useEffect, useRef, useState } from "react";
import HTMLFlipBook from "react-pageflip";
import { storage } from "../../services/apiServices";
import { FaChevronLeft, FaChevronRight, FaTimes, FaDownload } from "react-icons/fa";
import ImageLoader from "../ImageLoader";
import Borderbottom from "../Borderbottom";
import JSZip from "jszip";
import { saveAs } from "file-saver";

const FlipBook = () => {
  const [apiData, setApiData] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);

  const [activeYear, setActiveYear] = useState(null);
  const [activeMonth, setActiveMonth] = useState(null);
  const [activeDate, setActiveDate] = useState(null);
  const [selectedDateContent, setSelectedDateContent] = useState([]);

  const [lightboxOpen, setLightboxOpen] = useState(false);
  const [currentImageIndex, setCurrentImageIndex] = useState(0);

  const flipBookRef = useRef();
  const lightboxRef = useRef(null);

  useEffect(() => {
    fetch("https://codissia.com/codissia-backend/public/api/getAllYearsWithEvents")
      .then((res) => res.json())
      .then((data) => {
        const fetchedData = data.data.map((yearEntry) => {
          // Remove duplicate months if any
          const uniqueMonths = Array.from(
            new Map(
              yearEntry.months.map((month) => [month.month, month])
            ).values()
          );
          return { ...yearEntry, months: uniqueMonths };
        });

        setApiData(fetchedData);

        if (fetchedData.length > 0) {
          const defaultYear = fetchedData[0];
          const defaultMonth = defaultYear.months?.[0];
          const defaultEvent = defaultMonth?.events?.[0];

          setActiveYear(defaultYear.year);
          setActiveMonth(defaultMonth || null);
          setActiveDate(defaultEvent ? `${defaultMonth.month} ${defaultEvent.date}` : null);
          if (defaultEvent) {
            const parsedImages = parseImages(defaultEvent.image);
            setSelectedDateContent(parsedImages);
          }
        }
        setLoading(false);
      })
      .catch((err) => {
        console.error("API error:", err);
        setError("Failed to load data.");
        setLoading(false);
      });
  }, []);

  useEffect(() => {
    const handleClickOutside = (event) => {
      if (lightboxRef.current && !lightboxRef.current.contains(event.target)) {
        setLightboxOpen(false);
      }
    };

    if (lightboxOpen) {
      document.addEventListener("mousedown", handleClickOutside);
      document.body.style.overflow = "hidden";
    } else {
      document.body.style.overflow = "auto";
    }

    return () => {
      document.removeEventListener("mousedown", handleClickOutside);
      document.body.style.overflow = "auto";
    };
  }, [lightboxOpen]);

  const parseImages = (imageString) => {
    try {
      const images = JSON.parse(imageString);
      if (Array.isArray(images)) {
        return images;
      }
    } catch (e) {
      console.error("Error parsing images:", e);
    }
    return [];
  };

  const handleFlip = (direction) => {
    if (!flipBookRef.current) return;
    direction === "next"
      ? flipBookRef.current.pageFlip().flipNext()
      : flipBookRef.current.pageFlip().flipPrev();
  };

  const openLightbox = (index) => {
    setCurrentImageIndex(index);
    setLightboxOpen(true);
  };

  const navigateLightbox = (direction) => {
    if (direction === "prev") {
      setCurrentImageIndex((prev) =>
        prev === 0 ? selectedDateContent.length - 1 : prev - 1
      );
    } else {
      setCurrentImageIndex((prev) =>
        prev === selectedDateContent.length - 1 ? 0 : prev + 1
      );
    }
  };

  const handleDateClick = (event) => {
    setActiveDate(`${activeMonth.month} ${event.date}`);
    const parsedImages = parseImages(event.image);
    setSelectedDateContent(parsedImages);
  };

  const downloadAllImages = async () => {
    try {
      const zip = new JSZip();
      const imgFolder = zip.folder("images");

      const downloadPromises = selectedDateContent.map(async (imgPath, index) => {
        const response = await fetch(`${storage}${imgPath}`);
        const blob = await response.blob();
        const fileName = `image_${index + 1}.${blob.type.split('/')[1] || 'jpg'}`;
        imgFolder.file(fileName, blob);
      });

      await Promise.all(downloadPromises);

      const content = await zip.generateAsync({ type: "blob" });
      saveAs(content, `event_images_${activeYear}_${activeMonth?.month || ''}.zip`);
    } catch (error) {
      console.error("Error downloading images:", error);
      alert("Failed to download images. Please try again.");
    }
  };

  const downloadCurrentImage = async () => {
    try {
      const response = await fetch(`${storage}${selectedDateContent[currentImageIndex]}`);
      const blob = await response.blob();
      const fileName = `image_${currentImageIndex + 1}.${blob.type.split('/')[1] || 'jpg'}`;
      saveAs(blob, fileName);
    } catch (error) {
      console.error("Error downloading image:", error);
      alert("Failed to download image. Please try again.");
    }
  };

  const years = apiData.map((entry) => entry.year);
  const activeYearData = apiData.find((entry) => entry.year === activeYear);

  if (loading) return <ImageLoader />;
  if (error) return <div>{error}</div>;

  return (
    <section className="bulletin-section">
      <Borderbottom />
      <div className="container mt-5">
        <div className="row">
          <div className="col-md-6 mb-4 mb-md-0 d-flex justify-content-center flex-column bulletins-main-row">
            <div className="bulletins-main">
              {/* Year Stepper */}
              <div className="stepper-container">
                <h3>Select Year</h3>
                <div className="year-stepper">
                  {years.map((year) => (
                    <button
                      key={year}
                      className={`year-tab ${activeYear === year ? "active" : ""}`}
                      onClick={() => {
                        if (activeYear === year) return;

                        const selectedYear = apiData.find((y) => y.year === year);
                        const firstMonth = selectedYear?.months?.[0];
                        const firstEvent = firstMonth?.events?.[0];

                        setActiveYear(year);
                        setActiveMonth(firstMonth || null);
                        setActiveDate(firstEvent ? `${firstMonth.month} ${firstEvent.date}` : null);

                        if (firstEvent) {
                          const parsedImages = parseImages(firstEvent.image);
                          setSelectedDateContent(parsedImages);
                        } else {
                          setSelectedDateContent([]);
                        }
                      }}
                    >
                      {year}
                    </button>
                  ))}
                </div>
              </div>

              {/* Month Stepper */}
              {activeYear && (
                <div className="stepper-container">
                  <h3>{activeYear} - Select Month</h3>
                  <div className="month-stepper">
                    {activeYearData?.months.map((monthObj, idx) => (
                      <button
                        key={`${monthObj.month}-${idx}`}
                        className={`month-btn ${activeMonth?.month === monthObj.month ? "active" : ""}`}
                        onClick={() => {
                          if (activeMonth?.month === monthObj.month) return;

                          const firstEvent = monthObj?.events?.[0];
                          setActiveMonth(monthObj);
                          setActiveDate(firstEvent ? `${monthObj.month} ${firstEvent.date}` : null);

                          if (firstEvent) {
                            const parsedImages = parseImages(firstEvent.image);
                            setSelectedDateContent(parsedImages);
                          } else {
                            setSelectedDateContent([]);
                          }
                        }}
                      >
                        {monthObj.month}
                      </button>
                    ))}
                  </div>
                </div>
              )}

              {/* Date Buttons */}
              {activeMonth && (
                <div className="date-popup">
                  <h5>{activeMonth.month}, {activeYear} - Pick a Date</h5>
                  <div className="date-grid">
                    {activeMonth.events.map((event) => (
                      <button
                        key={event.id}
                        className={`date-btn ${activeDate === `${activeMonth.month} ${event.date}` ? "active" : ""}`}
                        onClick={() => handleDateClick(event)}
                      >
                        {activeMonth.month} {event.date}
                      </button>
                    ))}
                  </div>
                </div>
              )}
            </div>
          </div>

          {/* Flipbook Viewer */}
          <div className="col-md-6">
            <div className="book-container">
              <HTMLFlipBook
                ref={flipBookRef}
                width={350}
                height={500}
                size="stretch"
                minWidth={250}
                maxWidth={800}
                minHeight={400}
                maxHeight={1200}
                showCover={true}
                mobileScrollSupport={true}
                className="flip-book"
              >
                {selectedDateContent.map((imgPath, index) => (
                  <div key={index} className="page">
                    <div className="page-before"><div className="page-edge"></div></div>
                    <div className="page-content" onClick={() => openLightbox(index)}>
                      <img
                        src={`${storage}${imgPath}`}
                        alt="page content"
                        style={{
                          width: "100%",
                          height: "100%",
                          objectFit: "cover",
                          cursor: "pointer",
                        }}
                      />
                    </div>
                    <div className="page-after"><div className="page-edge"></div></div>
                  </div>
                ))}
              </HTMLFlipBook>
            </div>
          </div>
        </div>

        <div className="controls mb-4 text-center">
          <button onClick={() => handleFlip("prev")}>←</button>
          <button onClick={() => handleFlip("next")}>→</button>
        </div>
      </div>

      {/* Lightbox */}
      {lightboxOpen && (
        <div className="lightbox-overlay">
          <div className="lightbox-container" ref={lightboxRef}>
            <button className="lightbox-close" onClick={() => setLightboxOpen(false)}>
              <FaTimes />
            </button>
            <div className="lightbox-download-buttons">
              <button onClick={downloadCurrentImage}><FaDownload /> Current</button>
              <button onClick={downloadAllImages}><FaDownload /> All</button>
            </div>
            <button className="lightbox-nav prev" onClick={() => navigateLightbox("prev")}>
              <FaChevronLeft />
            </button>
            <div className="lightbox-content">
              <img src={`${storage}${selectedDateContent[currentImageIndex]}`} alt="Lightbox content" />
              <div className="lightbox-counter">
                {currentImageIndex + 1} / {selectedDateContent.length}
              </div>
            </div>
            <button className="lightbox-nav next" onClick={() => navigateLightbox("next")}>
              <FaChevronRight />
            </button>
          </div>
        </div>
      )}
    </section>
  );
};

export default React.memo(FlipBook);
