import React, { useState, useRef, useEffect, useContext } from "react";
import Masonry from "react-masonry-css";
import PropsContext from "../../context/PropsContext";

const GalleryInfo = () => {
  const [isOpen, setIsOpen] = useState(false);
  const [currentImage, setCurrentImage] = useState(0);
  const [loadedImages, setLoadedImages] = useState([]);
  const lightboxRef = useRef(null);
  const { gallery } = useContext(PropsContext);

  const breakpointColumnsObj = {
    default: 3,
    1100: 3,
    700: 2,
    500: 1,
  };

  const handleImageLoad = (id) => {
    setLoadedImages((prev) => [...prev, id]);
  };

  const openLightbox = (index) => {
    setCurrentImage(index);
    setIsOpen(true);
    document.body.style.overflow = "hidden";
  };

  const closeLightbox = () => {
    setIsOpen(false);
    document.body.style.overflow = "auto";
  };

  const goToPrevious = () => {
    setCurrentImage((prev) => (prev === 0 ? gallery.length - 1 : prev - 1));
  };

  const goToNext = () => {
    setCurrentImage((prev) => (prev === gallery.length - 1 ? 0 : prev + 1));
  };

  useEffect(() => {
    const handleKeyDown = (e) => {
      if (!isOpen) return;

      if (e.key === "Escape") {
        closeLightbox();
      } else if (e.key === "ArrowLeft") {
        goToPrevious();
      } else if (e.key === "ArrowRight") {
        goToNext();
      }
    };

    window.addEventListener("keydown", handleKeyDown);
    return () => window.removeEventListener("keydown", handleKeyDown);
  }, [isOpen, currentImage]);

  const handleLightboxClick = (e) => {
    if (e.target === lightboxRef.current) {
      closeLightbox();
    }
  };

  return (
    <section className="gallery-section">
      <div className="gallery-container">
        <Masonry
          breakpointCols={breakpointColumnsObj}
          className="masonry-grid"
          columnClassName="masonry-grid-column"
        >
          {gallery.map((image, index) => (
            <div
              key={image.id}
              className={`gallery-item ${loadedImages.includes(image.id) ? "loaded" : ""}`}
              onClick={() => openLightbox(index)}
            >
              <div className="image-wrapper">
                <img
                  src={image.image}
                  alt={image.name}
                  className="gallery-image"
                  loading="lazy"
                  onLoad={() => handleImageLoad(image.id)}
                />
                <div className="image-overlay">
                  <div className="overlay-content">
                    <h3 className="image-title">{image.name}</h3>
                  </div>
                </div>
              </div>
            </div>
          ))}
        </Masonry>

        {/* Lightbox */}
        {isOpen && (
          <div
            className="lightbox"
            ref={lightboxRef}
            onClick={handleLightboxClick}
          >
            
            <div className="lightbox-content">
              <button className="lightbox-nav prev" onClick={goToPrevious}>
                &#10094;
              </button>
              <img
                src={gallery[currentImage].image}
                alt={gallery[currentImage].name}
                className="lightbox-image"
              />
              <button className="lightbox-nav next" onClick={goToNext}>
                &#10095;
              </button>
              <div className="lightbox-caption">
                <h3>{gallery[currentImage].name}</h3>
                <p>
                  {currentImage + 1} / {gallery.length}
                </p>
              </div>
            </div>
          </div>
        )}
      </div>
    </section>
  );
};

export default  React.memo(GalleryInfo);
