import React, { useContext, useRef, useEffect, useState } from "react";
import { Swiper, SwiperSlide } from "swiper/react";
import { Navigation, Autoplay } from "swiper/modules";
import { storage } from "../../services/apiServices";
import PropsContext from "../../context/PropsContext";
import "swiper/css";
import "swiper/css/navigation";
import { SlCalender } from "react-icons/sl";
import Modal from "react-bootstrap/Modal";
import { FiChevronsRight } from "react-icons/fi";

const LatestUpdates = () => {
  const { events = [] } = useContext(PropsContext);
  const [selectedEvent, setSelectedEvent] = useState(null);
  const [showModal, setShowModal] = useState(false);
  const [currentImageIndex, setCurrentImageIndex] = useState(0);
  const swiperRef = useRef(null);

  useEffect(() => {
    if (swiperRef.current) {
      swiperRef.current.swiper.params.navigation.prevEl = ".custom-prev";
      swiperRef.current.swiper.params.navigation.nextEl = ".custom-next";
      swiperRef.current.swiper.navigation.init();
      swiperRef.current.swiper.navigation.update();
    }
  }, []);

  const handleReadMore = (event) => {
    setSelectedEvent(event);
    setCurrentImageIndex(0);
    setShowModal(true);
  };

  const closeModal = () => {
    setShowModal(false);
  };

  const parseImageArray = (imageString) => {
    try {
      if (Array.isArray(imageString)) return imageString;
      if (!imageString || imageString === "[]") return [];
      if (
        typeof imageString === "string" &&
        (imageString.startsWith("[") || imageString.startsWith("{"))
      ) {
        const parsed = JSON.parse(imageString);
        if (Array.isArray(parsed)) return parsed;
        if (typeof parsed === "object")
          return [parsed.url || parsed.image || parsed.path];
        return [parsed];
      }
      return [imageString];
    } catch (e) {
      console.error("Error parsing image array:", e);
      return [];
    }
  };

  const getImageUrl = (imagePath) => {
    try {
      if (!imagePath) throw new Error("No image path");
      if (/^(http|https|ftp):\/\//.test(imagePath)) return imagePath;
      const normalizedPath = imagePath.startsWith("/")
        ? imagePath.slice(1)
        : imagePath;
      return `${storage}${normalizedPath}`;
    } catch (error) {
      console.error("Error processing image URL:", error);
      return "";
    }
  };

  const formatDate = (dateString) => {
    if (!dateString) return "";
    const options = { year: "numeric", month: "long", day: "numeric" };
    return new Date(dateString).toLocaleDateString(undefined, options);
  };

  return (
    <section className="mt-5">
      <div className="container my-5 mt-5">
        <div className="section-title">
          <h2 className="text-center mb-4">LATEST UPDATES</h2>
        </div>

        <div className="row">
          {/* Left Column with Swiper Slider */}
          {/* <div className="col-lg-6 position-relative latest-updates-item mb-5">
            <button className="custom-prev btn">❮</button>
            <button className="custom-next btn">❯</button>

            <Swiper
              ref={swiperRef}
              spaceBetween={20}
              slidesPerView={1}
              loop={true}
              autoplay={{ delay: 1000, disableOnInteraction: false }}
              modules={[Navigation, Autoplay]}
              className="updates-swiper"
            >
              {events.map((update, index) => {
                if (!update) return null;
                const images = parseImageArray(update.image || "");
                const isEvent = update.category?.toLowerCase() === "event";

                return (
                  <SwiperSlide key={index}>
                    <div
                      className="latest-updates-info card p-3 shadow-sm"
                      data-aos="fade-up"
                    >
                      {images.length > 0 && (
                        <img
                          src={getImageUrl(images[0])}
                          alt={update.name || "Event image"}
                          className="card-img-top rounded"
                          style={{ height: "200px", objectFit: "cover" }}
                          onError={(e) => {
                            e.target.onerror = null;
                            e.target.src = "";
                            e.target.style.objectFit = "contain";
                          }}
                        />
                      )}

                      <div className="mt-3">
                        <h5 className="card-title mb-1">
                          {update.name || "Untitled Event"}
                        </h5>
                        {isEvent && (
                          <div className="text-muted small d-flex align-items-center lat-update-info">
                            {update.event_date_from && (
                              <>
                                <div className="d-flex align-items-center gap-1">
                                  <span>
                                    <SlCalender />
                                  </span>
                                  <p className="mb-0">
                                    {formatDate(update.event_date_from)}
                                  </p>
                                </div>
                                {update.event_date_to && (
                                  <>
                                    <span className="mx-2">|</span>
                                    <div className="d-flex align-items-center gap-1">
                                      <span>
                                        <SlCalender />
                                      </span>
                                      <p className="mb-0">
                                        {formatDate(update.event_date_to)}
                                      </p>
                                    </div>
                                  </>
                                )}
                              </>
                            )}
                          </div>
                        )}
                      </div>
                      <div className="mt-3">
                        <p
                          className="text-truncate"
                          style={{ maxHeight: "100px", overflow: "hidden" }}
                        >
                          {update.description || "No description available"}
                        </p>
                      </div>

                      {isEvent && update.registration_link && (
                        <div className="mt-2">
                          <span className="d-block register-link">
                            Registration link:
                            <a
                              href={update.registration_link}
                              target="_blank"
                              rel="noopener noreferrer"
                            >
                              {update.registration_link.substring(0, 30)}...
                            </a>
                          </span>
                        </div>
                      )}

                      <div className="mt-4">
                        <button
                          onClick={() => handleReadMore(update)}
                          className="read-more-updates"
                        >
                          Read More <span><FiChevronsRight/> </span>
                        </button>
                      </div>
                    </div>
                  </SwiperSlide>
                );
              })}
            </Swiper>
          </div> */}

          {/* Right Column with List Items */}
          {/* Carousel Slider for Events */}
          <div className="latest-updates-carousel position-relative">
            <button className="custom-prev btn">❮</button>
            <button className="custom-next btn">❯</button>

            <Swiper
              ref={swiperRef}
              spaceBetween={20}
              slidesPerView={3}
              loop={true}
              autoplay={{ delay: 5000, disableOnInteraction: false }}
              modules={[Navigation, Autoplay]}
              className="updates-swiper"
              breakpoints={{
                320: { slidesPerView: 1 },
                768: { slidesPerView: 2 },
                1200: { slidesPerView: 4 },
              }}
            >
              {events.map((update, index) => {
                if (!update) return null;
                const images = parseImageArray(update.image || "");
                const isEvent = update.category?.toLowerCase() === "event";

                return (
                  <SwiperSlide key={index}>
                    <div className="list-group-item h-100" data-aos="fade-up">
                      {images.length > 0 && (
                        <img
                          src={getImageUrl(images[0])}
                          alt={update.name || "Event image"}
                          className="card-img-top rounded"
                          style={{
                            // height: "350px",
                            objectFit: "cover",
                            aspectRatio: "3/2",
                          }}
                          onError={(e) => {
                            e.target.onerror = null;
                            e.target.src = "";
                            e.target.style.objectFit = "contain";
                          }}
                        />
                      )}
                      {isEvent && (
                        <div className="text-muted small d-flex align-items-center lat-update-info mt-2">
                          {update.event_date_from && (
                            <>
                              <div className="d-flex align-items-center gap-1">
                                <span>
                                  <SlCalender />
                                </span>
                                <p className="mb-0">
                                  {formatDate(update.event_date_from)}
                                </p>
                              </div>
                              {update.event_date_to && (
                                <>
                                  <span className="mx-2">|</span>
                                  <div className="d-flex align-items-center gap-1">
                                    <span>
                                      <SlCalender />
                                    </span>
                                    <p className="mb-0">
                                      {formatDate(update.event_date_to)}
                                    </p>
                                  </div>
                                </>
                              )}
                            </>
                          )}
                        </div>
                      )}
                      <div className="mt-3">
                        <p
                          className="list-group-item-des"
                          style={{ maxHeight: "80px", overflow: "hidden" }}
                        >
                          {update.description?.substring(0, 150) ||
                            "No description available"}
                          ...
                        </p>
                      </div>
                      {isEvent && update.registration_link && (
                        <div className="mt-2 registeration-link-main">
                          <h6>Registration link:</h6>
                          <div>
                            <a
                              href={update.registration_link}
                              target="_blank"
                              rel="noopener noreferrer"
                            >
                              {update.registration_link.substring(0, 30)}...
                            </a>
                          </div>
                        </div>
                      )}
                      <div className="mt-4">
                        <button
                          onClick={() => handleReadMore(update)}
                          className="read-more-updates"
                        >
                          Read More <FiChevronsRight />
                        </button>
                      </div>
                    </div>
                  </SwiperSlide>
                );
              })}
            </Swiper>
          </div>
        </div>
      </div>

      {/* Proper Bootstrap Modal */}
      <Modal
        show={showModal}
        onHide={closeModal}
        size="lg"
        centered
        dialogClassName="custom-event-modal"
      >
        {selectedEvent && (
          <>
            <Modal.Header closeButton></Modal.Header>
            <Modal.Body>
              <div className="row custom-modal-body">
                <div>
                  {" "}
                  <h4> {selectedEvent.name || "Untitled Event"} </h4>
                </div>
                <div className="col-md-12 image-section">
                  {parseImageArray(selectedEvent.image || "").length > 0 && (
                    <>
                      <div className="main-image-wrapper">
                        <img
                          src={getImageUrl(
                            parseImageArray(selectedEvent.image)[
                              currentImageIndex
                            ]
                          )}
                          alt={selectedEvent.name || "Event image"}
                          className="main-event-image"
                          onError={(e) => {
                            e.target.src = "";
                          }}
                        />
                      </div>
                      {parseImageArray(selectedEvent.image).length > 1 && (
                        <div className="thumbnail-gallery">
                          {parseImageArray(selectedEvent.image).map(
                            (img, idx) => (
                              <img
                                key={idx}
                                src={getImageUrl(img)}
                                alt={`${selectedEvent.name || "Event"} ${
                                  idx + 1
                                }`}
                                className={`thumbnail ${
                                  currentImageIndex === idx ? "active" : ""
                                }`}
                                onClick={() => setCurrentImageIndex(idx)}
                                onError={(e) => {
                                  e.target.src = "";
                                }}
                              />
                            )
                          )}
                        </div>
                      )}
                    </>
                  )}
                </div>
                <div className="col-md- content-section">
                  {selectedEvent.category?.toLowerCase() === "event" && (
                    <div className="event-dates">
                      {selectedEvent.event_date_from && (
                        <>
                          <div className="date-box">
                            <span>
                              <SlCalender />
                            </span>
                            <p>
                              <strong>From:</strong>{" "}
                              {formatDate(selectedEvent.event_date_from)}
                            </p>
                          </div>
                          {selectedEvent.event_date_to && (
                            <div className="date-box">
                              <span>
                                <SlCalender />
                              </span>
                              <p>
                                <strong>To:</strong>{" "}
                                {formatDate(selectedEvent.event_date_to)}
                              </p>
                            </div>
                          )}
                        </>
                      )}
                    </div>
                  )}
                  <div className="event-description">
                    <h5>Description</h5>
                    <p>
                      {selectedEvent.description || "No description available"}
                    </p>
                  </div>
                  {selectedEvent.place && (
                    <div className="event-location">
                      <h5>Locations</h5>
                      <p>{selectedEvent.place}</p>
                    </div>
                  )}
                </div>
              </div>
            </Modal.Body>
          </>
        )}
      </Modal>
    </section>
  );
};

export default React.memo(LatestUpdates);
