import React, {
  createContext,
  useState,
  useContext,
  useEffect,
  useMemo,
} from "react";
import apiService, { storage } from "../services/apiServices";
import api from "../services/apiServices";

const PropsContext = createContext();

export const PropsProvider = ({ children }) => {
  // UI State
  const [show, setShow] = useState(false);
  const [showNew, setShowNew] = useState(false);
  const [selectedItem, setSelectedItem] = useState(null);
  const [loadingmsg, setloadingmsg] = useState(false);

  // Props Data
  const [bannerData, setBannerData] = useState([]);
  const [bulletinsData, setBulletinsData] = useState([]);
  const [notificationsData, setNotificationsData] = useState([]);
  const [eventData, setEventData] = useState([]);
  const [galleryData, setGalleryData] = useState([]);
  const [committeeData, setCommitteeData] = useState([]);

  // Modal Handlers
  const handleShow = () => setShow(true);
  const handleClose = () => setShow(false);
  const handleShowDetails = (item) => {
    setSelectedItem(item);
    setShowNew(true);
  };
  const handleCloseDetails = () => setShowNew(false);

  // Initial Fetch
  useEffect(() => {
    fetchCriticalPropsData(); // Load essential props
    setTimeout(fetchLazyPropsData, 500); // Lazy load large props
  }, []);

  const fetchCriticalPropsData = async () => {
    try {
      const [bannerRes, bulletinRes, notificationRes] = await Promise.all([
        api.get("getbanner"),
        api.get("getAllYearsWithEvents"),
        api.get("getnotification"),
      ]);
      if (bannerRes?.status === 200) setBannerData(bannerRes.message);
      if (bulletinRes?.status === 200) setBulletinsData(bulletinRes.message);
      if (notificationRes?.status === 200)
        setNotificationsData(notificationRes.message);
    } catch (err) {
      console.error("Critical data fetch error:", err);
    }
  };

  const fetchLazyPropsData = async () => {
    try {
      const [eventsRes, galleryRes, committeeRes] = await Promise.all([
        api.get("getevents"),
        api.get("getgallery"),
        api.get("getexhibitors"),
      ]);
      if (eventsRes) setEventData(eventsRes.todays_events || []);
      if (galleryRes) setGalleryData(galleryRes.message || []);
      if (committeeRes) setCommitteeData(committeeRes.message || []);
    } catch (err) {
      console.error("Lazy data fetch error:", err);
    }
  };

  // Memoized Props Data
  const banners = useMemo(() =>
    bannerData.map((item) => ({
      ...item,
      image: item.image ? `${storage}/${item.image}` : "/images/default-banner.jpg",
    }))
  , [bannerData]);

  const bulletins = useMemo(() =>
    bulletinsData.map((item) => ({
      ...item,
      image: item.image ? `${storage}${item.image}` : "/images/default-bulletin.jpg",
    }))
  , [bulletinsData]);

  const notifications = useMemo(() =>
    notificationsData.map((item) => ({
      ...item,
      image: item.image ? `${storage}${item.image}` : "/images/default-notification.jpg",
    }))
  , [notificationsData]);

  const events = useMemo(() =>
    eventData.map((item) => ({
      ...item,
      image: item.image ? `${storage}${item.image}` : "",
    }))
  , [eventData]);

  const gallery = useMemo(() => {
    if (!Array.isArray(galleryData)) return [];
    return galleryData.map((item) => ({
      ...item,
      image: item.image ? `${storage}/${item.image}` : "/images/default-gallery.jpg",
      title: item.title || "No title available",
      description: item.description || "No description available",
    }));
  }, [galleryData]);

  const committee = useMemo(() =>
    committeeData.map((item) => ({
      ...item,
      image: item.image ? `${storage}${item.image}` : "/images/default-gallery.jpg",
    }))
  , [committeeData]);

  // Final Context
  const contextValue = useMemo(() => ({
    loadingmsg, setloadingmsg,
    handleShow, handleClose, handleShowDetails, handleCloseDetails,
    show, setShow, showNew, selectedItem,
    banners, bulletins, notifications, events, gallery, committee
  }), [
    loadingmsg, show, showNew, selectedItem,
    banners, bulletins, notifications, events, gallery, committee
  ]);

  return (
    <PropsContext.Provider value={contextValue}>
      {children}
    </PropsContext.Provider>
  );
};

export default PropsContext;
