import { Link, useLocation, useNavigate } from "react-router-dom";
import { useState, useEffect, useCallback } from "react";

export default function Navigation() {
  const { pathname } = useLocation();
  const navigate = useNavigate();
  const [openDropdown, setOpenDropdown] = useState(null); // State for which dropdown is open
  const [isMobileMenuOpen, setIsMobileMenuOpen] = useState(false); // State for mobile menu toggle

  // Close dropdowns and mobile menu on route change
  useEffect(() => {
    setOpenDropdown(null);
    setIsMobileMenuOpen(false);
  }, [pathname]);

  // Determine if a link is active
  const isActive = useCallback(
    (pathToCheck) => {
      if (Array.isArray(pathToCheck)) {
        return pathToCheck.includes(pathname);
      }
      return pathname === pathToCheck;
    },
    [pathname]
  );

  // Handle navigation and close menus/dropdowns
  const handleNavigation = useCallback(
    (path) => {
      navigate(path);
      setOpenDropdown(null); // Close any open dropdown
      setIsMobileMenuOpen(false); // Close mobile menu
      window.scrollTo(0, 0); // Scroll to top
    },
    [navigate]
  );

  // Toggle a specific dropdown
  const toggleDropdown = useCallback((index) => {
    setOpenDropdown((prev) => (prev === index ? null : index));
  }, []);

  // Toggle mobile menu
  const toggleMobileMenu = useCallback(() => {
    setIsMobileMenuOpen((prev) => !prev);
    setOpenDropdown(null); // Close any open dropdown when mobile menu toggles
  }, []);

  return (
    <div className="navbar-wrap main-menu">
      {/* Mobile menu toggle button (add this to your actual JSX for mobile) */}
      {/* Example: You might have a button like this in your header */}
      {/* <button className="mobile-menu-toggler" onClick={toggleMobileMenu}>
        <i className="fas fa-bars"></i>
      </button> */}

      <ul className={`navigation ${isMobileMenuOpen ? "mobile-open" : ""}`}>
        {/* Simple Links */}
        <li className={isActive("/") ? "ui-active-parent-1" : ""}>
          <Link to="/" onClick={() => handleNavigation("/")}>
            Home
          </Link>
        </li>
        <li className={isActive("/team") ? "ui-active-parent-1" : ""}>
          <Link to="/team" onClick={() => handleNavigation("/team")}>
            Team
          </Link>
        </li>

        {/* Programs Dropdown - Parent Navigates to /programs */}
        <li
          className={`menu-item-has-children ${
            openDropdown === 1 ||
            isActive(["/programs", "/announcements", "/events"])
              ? "open"
              : ""
          }`}
          onMouseEnter={() => setOpenDropdown(1)} // Desktop hover open
          onMouseLeave={() => setOpenDropdown(null)} // Desktop hover close
        >
          <Link
            to="/programs"
            className={
              isActive(["/programs", "/announcements", "/events"])
                ? "ui-active-parent-1"
                : ""
            }
            onClick={(e) => {
              // On desktop, clicking the link navigates directly.
              // On mobile, clicking the link should primarily toggle the dropdown for nested items.
              // We'll let the `dropdown-btn` handle the toggle explicitly for mobile.
              // For desktop, the default Link behavior is fine, or you can use handleNavigation.
              if (window.innerWidth <= 768) { // Example breakpoint for mobile
                e.preventDefault(); // Prevent navigation if on mobile and only want to open dropdown
                toggleDropdown(1);
              } else {
                handleNavigation("/programs");
              }
            }}
          >
            Programs
            <div
              className="dropdown-btn"
              onClick={(e) => {
                e.stopPropagation(); // Prevent the parent Link's onClick from firing
                toggleDropdown(1); // Explicitly toggle dropdown (good for mobile/touch)
              }}
            >
              <span className="fas fa-angle-down" />
            </div>
          </Link>
          <ul className="sub-menu">
            <li>
              <Link
                to="/announcements"
                className={isActive("/announcements") ? "ui-active-1" : ""}
                onClick={() => handleNavigation("/announcements")}
              >
                Announcements
              </Link>
            </li>
            <li>
              <Link
                to="/events"
                className={isActive("/events") ? "ui-active-1" : ""}
                onClick={() => handleNavigation("/events")}
              >
                Events
              </Link>
            </li>
          </ul>
        </li>

        {/* Simple Link */}
        <li className={isActive("/innovation") ? "ui-active-parent-1" : ""}>
          <Link to="/innovation" onClick={() => handleNavigation("/innovation")}>
            Innovation
          </Link>
        </li>

        {/* Incubation Dropdown - Parent Does NOT Navigate, Only Toggles */}
        <li
          className={`menu-item-has-children ${
            openDropdown === 2 || isActive(["/incubation", "/incubatees"])
              ? "open"
              : ""
          }`}
          onMouseEnter={() => setOpenDropdown(2)}
          onMouseLeave={() => setOpenDropdown(null)}
        >
          <Link
            to="#" // Link to '#' as this parent doesn't navigate directly
            className={
              isActive(["/incubation", "/incubatees"])
                ? "ui-active-parent-1"
                : ""
            }
            onClick={(e) => {
              e.preventDefault(); // Always prevent default for '#' links
              toggleDropdown(2); // Only toggles dropdown
            }}
          >
            Incubation
            <div
              className="dropdown-btn"
              onClick={(e) => {
                e.stopPropagation(); // Prevent the parent link's onClick from firing
                toggleDropdown(2); // Only toggle dropdown
              }}
            >
              <span className="fas fa-angle-down" />
            </div>
          </Link>
          <ul className="sub-menu">
            <li>
              <Link
                to="/incubation"
                className={isActive("/incubation") ? "ui-active-1" : ""}
                onClick={() => handleNavigation("/incubation")}
              >
                Incubation
              </Link>
            </li>
            <li>
              <Link
                to="/incubatees"
                className={isActive("/incubatees") ? "ui-active-1" : ""}
                onClick={() => handleNavigation("/incubatees")}
              >
                Incubatees
              </Link>
            </li>
          </ul>
        </li>

        {/* Simple Links */}
        <li className={isActive("/indigenisation") ? "ui-active-parent-1" : ""}>
          <Link
            to="/indigenisation"
            onClick={() => handleNavigation("/indigenisation")}
          >
            Indigenisation
          </Link>
        </li>
        <li className={isActive("/tenders") ? "ui-active-parent-1" : ""}>
          <Link to="/tenders" onClick={() => handleNavigation("/tenders")}>
            Tenders
          </Link>
        </li>

        {/* Gallery Dropdown - Parent Does NOT Navigate, Only Toggles */}
        <li
          className={`menu-item-has-children ${
            openDropdown === 3 || isActive(["/photogallery", "/newsgallery"])
              ? "open"
              : ""
          }`}
          onMouseEnter={() => setOpenDropdown(3)}
          onMouseLeave={() => setOpenDropdown(null)}
        >
          <Link
            to="#" // Link to '#' as this parent doesn't navigate directly
            className={
              isActive(["/photogallery", "/newsgallery"])
                ? "ui-active-parent-1"
                : ""
            }
            onClick={(e) => {
              e.preventDefault(); // Always prevent default for '#' links
              toggleDropdown(3); // Only toggles dropdown
            }}
          >
            Gallery
            <div
              className="dropdown-btn"
              onClick={(e) => {
                e.stopPropagation(); // Prevent the parent link's onClick from firing
                toggleDropdown(3); // Only toggle dropdown
              }}
            >
              <span className="fas fa-angle-down" />
            </div>
          </Link>
          <ul className="sub-menu">
            <li>
              <Link
                to="/photogallery"
                className={isActive("/photogallery") ? "ui-active-1" : ""}
                onClick={() => handleNavigation("/photogallery")}
              >
                Photo Gallery
              </Link>
            </li>
            <li>
              <Link
                to="/newsgallery"
                className={isActive("/newsgallery") ? "ui-active-1" : ""}
                onClick={() => handleNavigation("/newsgallery")}
              >
                News Gallery
              </Link>
            </li>
          </ul>
        </li>
        <li className={isActive("/contact") ? "ui-active-parent-1" : ""}>
          <Link to="/contact" onClick={() => handleNavigation("/contact")}>
            Contact
          </Link>
        </li>
      </ul>
    </div>
  );
}