import { useEffect, useRef, useState, useContext } from "react";
import { Swiper, SwiperSlide } from "swiper/react";
import { Navigation, Autoplay } from "swiper/modules";
import { FiChevronsRight, FiChevronsLeft } from "react-icons/fi";
import { FaCalendarAlt, FaCheckCircle } from "react-icons/fa";
import { PropsContext } from "../../../context/PropsContext";

import "swiper/css";
import "swiper/css/navigation";
import EventsSection from "../announcements/EventsSection";

const AnnouncementsSection = () => {
  const placeholderImage = "/images/home/announcements-1.png";
  const { formattedAnnouncements } = useContext(PropsContext);

  const [slidesPerView, setSlidesPerView] = useState(3);
  const [activeIndex, setActiveIndex] = useState(0);
  const prevRef = useRef(null);
  const nextRef = useRef(null);

  useEffect(() => {
    const updateSlidesPerView = () => {
      const width = window.innerWidth;
      setSlidesPerView(width < 768 ? 1 : width < 1024 ? 2 : 3);
    };
    updateSlidesPerView();
    window.addEventListener("resize", updateSlidesPerView);
    return () => window.removeEventListener("resize", updateSlidesPerView);
  }, []);

  // Format date to display
  const formatDate = (dateString) => {
    if (!dateString) return "Date not specified";
    try {
      const date = new Date(dateString);
      if (isNaN(date.getTime())) return "Invalid date";
      
      const options = { year: 'numeric', month: 'short', day: 'numeric' };
      return date.toLocaleDateString('en-US', options);
    } catch (error) {
      console.error("Error formatting date:", error);
      return "Invalid date";
    }
  };

  // Format the announcements data to match the component's structure
  const formattedData = formattedAnnouncements.map(item => ({
    id: item.id,
    title: item.title,
    content: item.description,
    image: item.image || placeholderImage,
    pdfUrl: item.pdfUrl || null,
    isActive: item.is_active === 1 || item.is_active === true,
    startDate: formatDate(item.start_date),
    endDate: formatDate(item.end_date),
    dateRange: item.start_date && item.end_date 
      ? `${formatDate(item.start_date)} - ${formatDate(item.end_date)}`
      : "Date range not specified"
  }));

  return (
    <div className="announcements-container">
      <div className="container">
        <div className="sec-title text-center news-letter mb-2">
          <h2>New Announcements</h2>
        </div>

        {/* Announcements Slider */}
        {formattedData.length > 0 && (
          <div className="announcements-slider-container mb-5">
            <div className="row align-items-center announcements-slider-row">
              <div className="swiper-nav-wrapper d-flex justify-content-between mt-2">
                <button ref={prevRef} className="swiper-button-prev-custom">
                  <FiChevronsLeft />
                </button>
                <button ref={nextRef} className="swiper-button-next-custom">
                  <FiChevronsRight />
                </button>
              </div>
              <div className="col-lg-1 col-md-1">

              </div>

              <div className="col-lg-7 col-md-9 col-sm-12 mb-5">
                <Swiper
                  modules={[Navigation, Autoplay]}
                  navigation={{
                    prevEl: prevRef.current,
                    nextEl: nextRef.current,
                  }}
                  onBeforeInit={(swiper) => {
                    swiper.params.navigation.prevEl = prevRef.current;
                    swiper.params.navigation.nextEl = nextRef.current;
                  }}
                  onSlideChange={(swiper) => setActiveIndex(swiper.realIndex)}
                  autoplay={{ delay: 3000 }}
                  loop
                  className="announcement-swiper"
                >
                  {formattedData.map((item, index) => (
                    <SwiperSlide key={item.id}>
                      <div className="image-container">
                        <img
                          src={item.image}
                          alt={item.title}
                          className="img-fluid w-100"
                        />
                      </div>
                    </SwiperSlide>
                  ))}
                </Swiper>
              </div>

              <div className="col-lg-4 col-md-4 col-sm-12">
                <div className="content-container">
                  <h3>{formattedData[activeIndex]?.title || "No title available"}</h3>
                  <div className="announcement-meta mb-3">
                    <span className="date-range me-3">
                      <FaCalendarAlt className="me-2" />
                      {formattedData[activeIndex]?.dateRange || "Date not specified"}
                    </span>
                    {formattedData[activeIndex]?.isActive && (
                      <span className="status-badge active">
                        <FaCheckCircle className="me-1" />
                        Active
                      </span>
                    )}
                  </div>
                  <p>{formattedData[activeIndex]?.content || "No content available"}</p>
                  {formattedData[activeIndex]?.pdfUrl && (
                    <div className="btn-download">
                      <a
                        href={formattedData[activeIndex].pdfUrl}
                        className="read-more d-inline-flex align-items-center"
                        target="_blank"
                        rel="noopener noreferrer"
                      >
                        Download PDF{" "}
                        <span className="ps-2">
                          <FiChevronsRight />
                        </span>
                      </a>
                    </div>
                  )}
                </div>
              </div>
            </div>
          </div>
        )}

        {formattedData.length === 0 && (
          <div className="text-center py-5">
            <p>No announcements available at the moment.</p>
          </div>
        )}
      </div>
      <EventsSection />
    </div>
  );
};

export default AnnouncementsSection;