import React, { useContext, useEffect, useState } from "react";
import { PropsContext } from "../../../context/PropsContext";
import { RiNewspaperLine } from "react-icons/ri";

const NewsletterSchedule = () => {
  const { apiService, storage } = useContext(PropsContext);
  const [newsletters, setNewsletters] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);

  useEffect(() => {
    const fetchNewsletters = async () => {
      try {
        const response = await apiService.get("newsletter");
        console.log("API Response:", response); // Debug log
        
        // Check both response.data.data and response.data
        const data = response.data?.data || response.data;
        
        if (data) {
          setNewsletters(Array.isArray(data) ? data : [data]);
        } else {
          setNewsletters([]);
        }
      } catch (err) {
        console.error("Failed to fetch newsletters:", err);
        setError("Failed to load newsletters. Please try again later.");
      } finally {
        setLoading(false);
      }
    };

    fetchNewsletters();
  }, [apiService]);

  if (loading) {
    return (
      <section className="newsletter-section sec-ptb">
        <div className="container">
          <div className="text-center py-5">
            <div className="spinner-border text-primary" role="status">
              <span className="visually-hidden">Loading...</span>
            </div>
          </div>
        </div>
      </section>
    );
  }

  if (error) {
    return (
      <section className="newsletter-section sec-ptb">
        <div className="container">
          <div className="alert alert-danger">{error}</div>
        </div>
      </section>
    );
  }

  return (
    <section className="newsletter-section sec-ptb">
      <div className="container">
        <div className="sec-content text-center mb-5">
          <h6 className="sec-sub-title">Our Newsletter</h6>
          <h2 className="teamx-title">Explore Our latest Newsletter</h2>
        </div>
        
        {newsletters.length === 0 ? (
          <div className="text-center py-5">
            <p>No newsletters available at the moment.</p>
          </div>
        ) : (
          <div className="row gy-4">
            {newsletters.map((item, index) => (
              <div className="col-lg-4 col-md-6" key={index}>
                <div className="newsletter-card shadow-sm h-100">
                  <div className="newsletter-body">
                    <h5 className="newsletter-title">{item.title || "Untitled Newsletter"}</h5>
                    <p className="newsletter-date">
                      {new Date(item.start_date).toLocaleDateString("en-US", {
                        year: "numeric",
                        month: "short",
                        day: "numeric",
                      })}
                    </p>
                    <div className="btn-download">
                      <a 
                        href={item.pdf_path ? `${storage}/${item.pdf_path}` : "#"} 
                        download
                        target="_blank"
                        rel="noopener noreferrer"
                      >
                        Download
                      </a>
                    </div>
                    <div className="icon-newsletter">
                      <RiNewspaperLine />
                    </div>
                  </div>
                </div>
              </div>
            ))}
          </div>
        )}
      </div>
    </section>
  );
};

export default NewsletterSchedule;