<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Enquiry;
use App\Mail\EnquiryReceived;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Hash;
use App\Models\Member;

class ApiController extends Controller
{


    public function enquiry(Request $request)
    {
        try {
            // Validate the request data
            $validatedData = $request->validate([
                'name' => 'required|string|max:255',
                'email' => 'required|email|max:255',
                'mobile' => 'required|numeric|digits_between:10,15',
                'subject' => 'nullable|string',
                'message' => 'nullable|string',
                'address' => 'nullable|string',
            ]);

            // Store the enquiry in the database
            $enquiry = Enquiry::create($validatedData);

            // Send the email
            Mail::to('madhumitha@clouddreams.in')->send(new EnquiryReceived($enquiry)); // Send the email to the admin

            // Return a success response
            return response()->json([
                'success' => true,
                'message' => 'Enquiry submitted successfully!',
            ], 200);
        } catch (\Illuminate\Validation\ValidationException $e) {
            // Return validation errors
            return response()->json([
                'success' => false,
                'message' => 'Validation errors occurred.',
                'errors' => $e->errors(),
            ], 422);
        } catch (\Exception $e) {
            // Handle other errors
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while submitting the enquiry.',
                'errors' => $e->getMessage(),
            ], 500);
        }
    }

    public function profile(Request $request)
    {
        $admin = Auth::user();
        return view('auth.profile', compact('admin'));
    }

    public function updateProfile(Request $request)
    {

        // Validate the incoming data
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email,' . Auth::id(),
            'password' => 'nullable|string|min:8|confirmed',  // Password is optional
        ]);

        // Get the currently authenticated admin user
        $admin = Auth::user();

        // Update name and email
        $admin->name = $request->input('name');
        $admin->email = $request->input('email');

        // If password is provided, hash and update it
        if ($request->filled('password')) {
            $admin->password = Hash::make($request->input('password'));
        }
        if ($request->hasFile('image')) {
            // Get the file from the request
            $file = $request->file('image');

            // Generate a unique filename for the file
            $filename = uniqid() . '.' . $file->getClientOriginalExtension();

            // Move the file to the 'public/assets/profile' directory
            $file->move(public_path('assets/profile'), $filename);

            // Set the profile URL to the relative path
            $admin->profile_url = 'assets/profile/' . $filename;
        }


        // Save the updated profile
        $admin->save();

        // Redirect back with a success message
        return redirect()->route('profile')->with('success', 'Profile updated successfully');
    }

    public function changepassword(Request $request)
    {

        // Validate the incoming data
        // $request->validate([
        //     'password' => 'nullable|string|confirmed',  // Password is optional
        // ]);

        // Get the currently authenticated admin user
        $admin = Auth::user();


        // If password is provided, hash and update it
        if ($request->filled('password')) {

            $admin->password = Hash::make($request->input('password'));
        }
        // Save the updated profile
        $admin->save();

        // Redirect back with a success message
        return redirect()->route('profile')->with('success', 'Profile updated successfully');
    }

    public function editMember($id)
    {
        // Find the member by ID
        $member = Member::find($id);

        // Check if the member exists
        if (!$member) {
            return response()->json([
                'success' => false,
                'message' => 'Member not found',
            ], 404);
        }

        // Return the member data
        return response()->json([
            'success' => true,
            'data' => $member,
        ]);
    }
}
