<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Bulletin;
use Illuminate\Support\Facades\Validator;

class BulletinController extends Controller
{
    public function index()
    {
        $bulletin = Bulletin::paginate(10);
        return view('layouts.Bulletin.index', compact('bulletin'));
    }


    public function create()
    {
        return view('layouts.bulletin.add');
    }

    // Store a new Bulletin
    public function store(Request $request)
    {

        // Validate the input data
        $validator = Validator::make($request->all(), [
            'name' => 'nullable|string',
            'date' => 'required|integer',
            'month' => 'required|integer',
            'year' => 'required|integer',
            'image' => $request->id ? 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048' : 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'errors' => $validator->errors(),
            ], 422);
        }

        // Handle the image upload
        $imagePath = $request->id ? Bulletin::find($request->id)->image : null;

        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = time() . '.' . $image->getClientOriginalExtension();

            try {
                $image->move(public_path('images'), $imageName);
                $imagePath = 'images/' . $imageName;
            } catch (\Exception $e) {
                return response()->json(['error' => 'Failed to upload image.'], 500);
            }
        }

        // Save the event data (including the image path)
        Bulletin::updateOrCreate(
            ['id' => $request->id],
            [
                'name' => $request->name,
                'date' => $request->date,
                'month' => $request->month,
                'year' => $request->year,
                'image' => $imagePath,
            ]
        );

        return response()->json([
            'success' => true,
            'redirect_url' => route('bulletin.index'),
        ]);
    }
    // Update an existing Bulletin
    public function edit($id)
    {
        $bulletin = Bulletin::findOrFail($id);

        return view('layouts.bulletin.edit', compact('bulletin')); // Return form view
    }

    // Update an existing Bulletin

    // Delete a Bulletin
    public function destroy($id)
    {
        $bulletin = Bulletin::findOrFail($id);
        $bulletin->delete();
        $bulletin = Bulletin::paginate(10);

        return view('layouts.bulletin.index', compact('bulletin'));
    }

    public function getAllYearsWithEvents()
    {
        // Fetch all events, ordered by year, month, and date
        $events = Bulletin::orderBy('year', 'DESC')
            ->orderBy('month', 'DESC')
            ->orderBy('date', 'DESC')
            ->get();

        // Group events by year
        $groupedByYear = $events->groupBy('year');

        // Format the response
        $formattedYears = [];
        foreach ($groupedByYear as $year => $yearEvents) {
            $formattedYears[] = [
                'year' => $year,
                'months' => $yearEvents->groupBy('month')->map(function ($monthEvents, $month) {
                    return [
                        'month' => \Carbon\Carbon::createFromFormat('m', $month)->format('F'), // Convert month number to name
                        'events' => $monthEvents->map(function ($event) {
                            return [
                                'id' => $event->id,
                                'name' => $event->name,
                                'description' => $event->description,
                                'date' => $event->date, // Keep the day of the month
                            ];
                        })->values(),
                    ];
                })->values(),
            ];
        }

        return response()->json(['data' => $formattedYears], 200);
    }

    public function getEventsByDate(Request $request)
    {
        // Validate the incoming request parameters
        $validated = $request->validate([
            'year' => 'required|integer|min:1900|max:2100',
            'month' => 'required|integer|min:1|max:12',
            'date' => 'nullable|integer|min:1|max:31',
        ]);

        // Query the database based on year, month, and optional date
        $query = Bulletin::where('year', $validated['year'])
            ->where('month', $validated['month']);

        if (!empty($validated['date'])) {
            $query->where('date', $validated['date']);
        }

        $events = $query->get();

        // Check if events exist
        if ($events->isEmpty()) {
            return response()->json(['message' => 'No events found for the given date.'], 404);
        }

        // Format the response
        $formattedEvents = $events->map(function ($event) {
            return [
                'id' => $event->id,
                'name' => $event->name,
                'description' => $event->description,
                'date' => $event->date,
                'month' => $event->month,
                'year' => $event->year,
                'image' => $event->image,
            ];
        });

        return response()->json(['data' => $formattedEvents], 200);
    }
}
