<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Events;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;


class EventController extends Controller
{
    public function index()
    {
        $events = Events::paginate(10);
        return view('layouts.event.index', compact('events'));
    }


    public function create()
    {
        return view('layouts.event.add');
    }

    // Store a new Events
    public function store(Request $request)
    {

        // Validate the input data
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'event_date_from' => 'required|date',
            'event_date_to' => 'required|date|after_or_equal:event_date_from',
            'event_time_from' => 'required',
            'event_time_to' => 'required',

        ]);

        if ($validator->fails()) {
            return response()->json([
                'errors' => $validator->errors()
            ], 422);
        }
        // Handle the image upload if the file exists
        $imagePath = null;
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            // Generate a unique file name
            $imageName = time() . '.' . $image->getClientOriginalExtension();
            // Move the image to the public/images directory
            $image->move(public_path('images'), $imageName);
            // Save the image path to be stored in the database
            $imagePath = 'images/' . $imageName;
        }

        // Save the event data (including the image path)
        Events::updateOrCreate(
            ['id' => $request->id],
            [
                'name' => $request->name,
                'event_date_from' => $request->event_date_from,
                'event_date_to' => $request->event_date_to,
                'event_time_from' => $request->event_time_from,
                'event_time_to' => $request->event_time_to,
                'place' => $request->place,
                'capacity' => $request->capacity,
                'organizer' => $request->organizer,
                'registration_link' => $request->registration_link,
                'image' => $imagePath,  // Store the image path in the database
            ]
        );

        return response()->json([
            'success' => true,
            'redirect_url' => route('event.index'),
        ]);
    }


    // Update an existing Events
    public function edit($id)
    {
        $event = Events::findOrFail($id);

        return view('layouts.event.edit', compact('event')); // Return form view
    }

    // Update an existing Events


    // Delete a Events
    public function destroy($id)
    {
        $event = Events::findOrFail($id);
        $event->delete();
        $event = Events::paginate(10);
        return view('layouts.event.index', compact('event'));
    }


    public function getEvents()
    {
        // Current date
        $today = Carbon::today();

        // Fetch all events from the database
        $events = Events::all();

        // Categorize events
        $upcomingEvents = $events->filter(fn($event) => Carbon::parse($event->date)->isFuture());
        $pastEvents = $events->filter(fn($event) => Carbon::parse($event->date)->isPast());
        $todaysEvents = $events->filter(fn($event) => Carbon::parse($event->date)->isToday());

        // Prepare the response
        return response()->json([
            'todays_events' => $todaysEvents->values(),
            'upcoming_events' => $upcomingEvents->values(),
            'past_events' => $pastEvents->values(),
        ]);
    }
}
