<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Exhibitors;
use Illuminate\Support\Facades\Validator;

class ExhibitorController extends Controller
{
    public function index()
    {
        $exhibitors = Exhibitors::paginate(10);
        return view('layouts.exhibitors.index', compact('exhibitors'));
    }


    public function create()
    {
        return view('layouts.exhibitors.add');
    }

    // Store a new Exhibitors
    public function store(Request $request)
    {
        // Validate the input data
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'event_date' => 'required|date',
            'description' => 'nullable|string|max:255',
            'status' => 'nullable|boolean',
            'image' => $request->id ? 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048' : 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'errors' => $validator->errors(),
            ], 422);
        }

        // Get the existing exhibitor if it exists, or set imagePath to null
        $exhibitor = $request->id ? Exhibitors::find($request->id) : null;

        $imagePath = $exhibitor ? $exhibitor->image : null;

        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = time() . '.' . $image->getClientOriginalExtension();

            try {
                $image->move(public_path('images'), $imageName);
                $imagePath = 'images/' . $imageName;
            } catch (\Exception $e) {
                return response()->json(['error' => 'Failed to upload image.'], 500);
            }
        }

        // Save the event data (including the image path)
        Exhibitors::updateOrCreate(
            ['id' => $request->id],
            [
                'name' => $request->name,
                'image' => $imagePath,
                'link' => $request->link,
                'event_date' => $request->event_date,

            ]
        );

        return response()->json([
            'success' => true,
            'redirect_url' => route('exhibitors.index'),
        ]);
    }

    // Update an existing Exhibitors
    public function edit($id)
    {
        $exhibitors = Exhibitors::findOrFail($id);

        return view('layouts.exhibitors.edit', compact('exhibitors')); // Return form view
    }

    // Update an existing Exhibitors

    // Delete a Exhibitors
    public function destroy($id)
    {
        $exhibitors = Exhibitors::findOrFail($id);
        $exhibitors->delete();
        $exhibitors = Exhibitors::paginate(10);

        return view('layouts.exhibitors.index', compact('exhibitors'));
    }

    public function exhibitors(Request $request)
    {
        // Fetch all galleries
        $exhibitors = Exhibitors::all();

        // Check if the collection is empty
        if ($exhibitors->isEmpty()) {
            return response()->json([
                'status' => 404,
                'message' => 'No exhibitors image found!'
            ]);
        }

        // Return gallery data if found
        return response()->json([
            'status' => 200,
            'message' => $exhibitors
        ]);
    }
}
