<?php

namespace App\Http\Controllers;

use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\Member;
use Illuminate\Validation\Rule;
use App\Imports\MembersImport;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Auth;

class MemberController extends Controller
{
    public function index()
    {
        $members = Member::paginate(10); // Fetch 10 members per page
        return view('layouts.member.list', compact('members'));
    }

    // Show form to create a new Member
    public function create()
    {
        return view('layouts.member.add'); // Return form view
    }

    // Store a new Member
    public function store(Request $request)
    {

        $request->validate([
            'name' => [
                'required',
                'string',
                'max:255',
            ],
            'company_name' => [
                'required',
                'string',
                'max:255',
                Rule::unique('members')->ignore($request->id),
            ],
            'email' => [
                'required',
                'email',
                'max:255',
                Rule::unique('members')->ignore($request->id),
            ],
            'phone_number' => [
                'required',
                'string',
                Rule::unique('members')->ignore($request->id),
            ],
            'category' => [
                'required',
                'string',

            ]
        ], [
            'name.required' => 'Name is required.',
            'company_name.required' => 'Company name is required.',
            'company_name.unique' => 'Company name has already been taken.',
            'email.required' => 'Email is required.',
            'category.required' => 'Category is required.',
            'email.unique' => 'Email has already been taken.',
            'phone_number.required' => 'Phone number is required.',
            'phone_number.unique' => 'Phone number has already been taken.',
        ]);

        // Ensure required fields are present before calling updateOrCreate

        // Use updateOrCreate to update if id is present, otherwise create a new Member
        Member::updateOrCreate(
            ['id' => $request->id], // Condition to find the existing record by ID
            [
                'name' => $request->name,
                'company_name' => $request->company_name,
                'category' => $request->category,
                'email' => $request->email ?? null,  // Ensure email is present
                'phone_number' => $request->phone_number, // Ensure phone number is present
                'address' => $request->address,
                'designation' => $request->designation,
                'code_no' => $request->code_no ?? null,
                'website_url' => $request->website_url,
                'keywords' => $request->keywords,
            ]
        );

        return response()->json([
            'success' => true,
            'redirect_url' => route('members.index'), // Dynamically generate the URL
        ]);
    }





    // Show form to edit an existing Member
    public function edit($id)
    {
        $member = Member::findOrFail($id); // Find Member or fail

        return view('layouts.member.edit', compact('member')); // Return form view
    }

    // Update an existing Member


    // Delete a Member
    public function destroy($id)
    {
        $Member = Member::findOrFail($id); // Find Member or fail
        $Member->delete();

        // Return the updated list of categories as a JSON response
        $members = Member::all();

        return view('layouts.member.list', compact('members'));
    }
    public function import(Request $request)
    {

        $request->validate([
            'import_file' => 'required|file|mimes:xlsx,csv',
        ]);

        Excel::import(new MembersImport, $request->file('import_file'));

        return redirect()->back()->with('success', 'Data imported successfully.');
    }

    public function member_view()
    {
        $members = Member::all(); // Fetch all categories
        $randomToken = Str::random(60);
        return response()->json([
            'status' => 'success',
            'data' => $members,
            'access_token' => $randomToken
        ]);
    }


    public function search_members(Request $request)
    {

        // Get search query from the request
        $query = $request->input('query');

        // Fetch members based on the query
        $members = Member::where('name', 'LIKE', "%{$query}%")
            ->orWhere('company_name', 'LIKE', "%{$query}%")
            ->orWhere('category', 'LIKE', "%{$query}%")
            ->get();

        // Return the filtered members and token
        if (empty($members)) {
            return response()->json([
                'status' => '400',
                'data' => "No data found",

            ]);
        } else {
            return response()->json([
                'status' => '200',
                'data' => $members,

            ]);
        }
    }

    public function alphabet(Request $request)
    {
        // Get the alphabet from the request
        $alphabet = $request->input('alphabet');

        // Fetch members whose name starts with the provided alphabet
        $members = Member::where('name', 'LIKE', "{$alphabet}%")->get();

        // Check if the collection is empty
        if (!$members->isEmpty()) {
            return response()->json([
                'status' => '200',
                'data' => $members,
            ]);
        } else {
            return response()->json([
                'status' => '400',
                'data' => "No data found",
            ]);
        }
    }

    public function allmembers(Request $request)
    {
        $members = Member::paginate(10);

        // Check if the collection is empty
        if (!$members->isEmpty()) {
            return response()->json([
                'status' => '200',
                'data' => $members,
            ]);
        } else {
            return response()->json([
                'status' => '400',
                'data' => "No data found",
            ]);
        }
    }
}
