<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Notification;
use Illuminate\Support\Facades\Validator;

class NotificationController extends Controller
{
    public function index()
    {
        $notification = Notification::paginate(10);
        return view('layouts.notification.index', compact('notification'));
    }


    public function create()
    {
        return view('layouts.notification.add');
    }

    // Store a new Notification
    public function store(Request $request)
    {
        
        // Validate the input data
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'date' => 'required|date',
            'description' => 'nullable|string|max:255',
            'status' => 'nullable|boolean',
            'image' => $request->id ? 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048' : 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'errors' => $validator->errors(),
            ], 422);
        }

        // Handle the image upload
        $imagePath = $request->id ? Notification::find($request->id)->image : null;

        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = time() . '.' . $image->getClientOriginalExtension();

            try {
                $image->move(public_path('images'), $imageName);
                $imagePath = 'images/' . $imageName;
            } catch (\Exception $e) {
                return response()->json(['error' => 'Failed to upload image.'], 500);
            }
        }

        // Save the event data (including the image path)
        Notification::updateOrCreate(
            ['id' => $request->id],
            [
                'name' => $request->name,
                'image' => $imagePath,
                'date' => $request->date,
                'description' => $request->description,
                'status' => $request->status,
                'image'=> $imagePath
            ]
        );

        return response()->json([
            'success' => true,
            'redirect_url' => route('notification.index'),
        ]);
    }
    // Update an existing Notification
    public function edit($id)
    {
        $notification = Notification::findOrFail($id);

        return view('layouts.notification.edit', compact('notification')); // Return form view
    }

    // Update an existing Notification

    // Delete a Notification
    public function destroy($id)
    {
        $notification = Notification::findOrFail($id);
        $notification->delete();
        $notification = Notification::paginate(10);

        return view('layouts.notification.index', compact('notification'));
    }

    public function notification(Request $request)
    {
        // Fetch all galleries
        $notification = Notification::all();

        // Check if the collection is empty
        if ($notification->isEmpty()) {
            return response()->json([
                'status' => 404,
                'message' => 'No notification found!'
            ]);
        }

        // Return gallery data if found
        return response()->json([
            'status' => 200,
            'message' => $notification
        ]);
    }
}
